/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2015 Frederik Heber. All rights reserved.
 *
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * QtMoleculeItem.cpp
 *
 *  Created on: Jan 17, 2015
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "QtMoleculeItem.hpp"

#include <QModelIndex>

#include "UIElements/Views/Qt4/MoleculeList/QtMoleculeList.hpp"

#include "CodePatterns/MemDebug.hpp"

#include "CodePatterns/Assert.hpp"
#include "CodePatterns/toString.hpp"

#include "CodePatterns/Observer/Notification.hpp"

#include <algorithm>

#include "Descriptors/MoleculeIdDescriptor.hpp"
#include "molecule.hpp"
#include "World.hpp"

QtMoleculeItem::QtMoleculeItem(
     const moleculeId_t _molid,
     const channellist_t &_channellist,
     const enum MoveTypes _movetype,
     const emitDirtyState_t _emitDirtyState) :
  Observer("QtMoleculeItem"),
  molid(_molid),
  movetype(_movetype),
  channellist(_channellist),
  IsSignedOn(false),
  dirty(true),
  emitDirtyState(_emitDirtyState)
{
  signOnToMolecule();
  setFlags(flags() | Qt::ItemIsSelectable);
}

void QtMoleculeItem::signOnToMolecule()
{
  const molecule * const mol = const_cast<const World &>(World::getInstance()).
      getMolecule(MoleculeById(molid));
  if (mol != NULL) {
    if (!IsSignedOn)
      for (channellist_t::const_iterator channeliter = channellist.begin();
          channeliter != channellist.end(); ++channeliter)
        mol->signOn(this, *channeliter);
    IsSignedOn = true;
  }
}

void QtMoleculeItem::signOffFromMolecule()
{
  const molecule * const mol = const_cast<const World &>(World::getInstance()).
      getMolecule(MoleculeById(molid));
  if (mol != NULL) {
    if (IsSignedOn)
      for (channellist_t::const_iterator channeliter = channellist.begin();
          channeliter != channellist.end(); ++channeliter)
        mol->signOff(this, *channeliter);
  }
  IsSignedOn = false;
}

QtMoleculeItem::~QtMoleculeItem()
{
  signOffFromMolecule();
}

void QtMoleculeItem::updateState()
{
  if (dirty) {
    internal_updateState();
    dirty = false;
  }
}

void QtMoleculeItem::update(Observable *publisher)
{}

void QtMoleculeItem::recieveNotification(Observable *publisher, Notification_ptr notification)
{
  if (dynamic_cast<molecule *>(publisher) != NULL) {
    if (notification->getChannelNo() == molecule::AboutToBeRemoved) {
      signOffFromMolecule();
      // prevent any remaining updates from accessing the molecule
      //molid = -1;
      dirty = false;
    } else {
      channellist_t::const_iterator iter =
          std::find(channellist.begin(), channellist.end(), notification->getChannelNo());
      if (iter != channellist.end()) {
        dirty = true;
        emitDirtyState(this, movetype);
      } else {
        ASSERT(0,
            "QtMoleculeItem::recieveNotification() - received notification to channel "
            +toString(notification->getChannelNo())+" we are not subscribed to.");
      }
    }
  }
}

void QtMoleculeItem::subjectKilled(Observable *publisher)
{
  IsSignedOn = false;
  dirty = false;
}

const molecule * const QtMoleculeItem::getMolecule() const
{
  const molecule * const mol = const_cast<const World &>(World::getInstance()).getMolecule(MoleculeById(molid));
  ASSERT (mol != NULL,
      "QtMoleculeItem::getMolecule() - mol with "+toString(molid)+" is gone.");
  return mol;
}
