/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * SubgraphDissectionAction.cpp
 *
 *  Created on: May 9, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "Descriptors/AtomIdDescriptor.hpp"
#include "Descriptors/MoleculeDescriptor.hpp"

#include "atom.hpp"
#include "bond.hpp"
#include "Graph/BondGraph.hpp"
#include "config.hpp"
#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"
#include "molecule.hpp"
#include "World.hpp"

#include <iostream>
#include <string>

typedef std::map< moleculeId_t, std::vector<atomId_t> > MolAtomList;
typedef std::map< atomId_t, atomId_t > AtomAtomList;

using namespace std;

#include "Actions/FragmentationAction/SubgraphDissectionAction.hpp"

// and construct the stuff
#include "SubgraphDissectionAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
Action::state_ptr FragmentationSubgraphDissectionAction::performCall() {
  // obtain information
  getParametersfromValueStorage();


  // first create stuff for undo state
  LOG(0, "STATUS: Creating undo state.");
  MolAtomList moleculelist;
  vector<molecule *> allmolecules = World::getInstance().getAllMolecules();
  for (vector<molecule *>::const_iterator moliter = allmolecules.begin(); moliter != allmolecules.end(); ++moliter) {
    std::vector<atomId_t> atomlist;
    atomlist.resize((*moliter)->size());
    for (molecule::const_iterator atomiter = (*moliter)->begin(); atomiter != (*moliter)->end(); ++atomiter) {
      atomlist.push_back((*atomiter)->getId());
    }
    moleculelist.insert( std::pair< moleculeId_t, std::vector<atomId_t> > ((*moliter)->getId(), atomlist));
  }
  FragmentationSubgraphDissectionState *UndoState = new FragmentationSubgraphDissectionState(moleculelist, params);

  // 0a. remove all present molecules
  LOG(0, "STATUS: Removing all present molecules.");
  MoleculeListClass *molecules = World::getInstance().getMolecules();
  for (vector<molecule *>::iterator MolRunner = allmolecules.begin(); MolRunner != allmolecules.end(); ++MolRunner) {
    molecules->erase(*MolRunner);
    World::getInstance().destroyMolecule(*MolRunner);
  }

  // 0b. remove all bonds and construct a molecule with all atoms
  molecule *mol = World::getInstance().createMolecule();
  {
    vector <atom *> allatoms = World::getInstance().getAllAtoms();
    for(vector<atom *>::iterator AtomRunner = allatoms.begin(); AtomRunner != allatoms.end(); ++AtomRunner) {
//      const BondList& ListOfBonds = (*AtomRunner)->getListOfBonds();
//      for(BondList::iterator BondRunner = ListOfBonds.begin();
//          !ListOfBonds.empty();
//          BondRunner = ListOfBonds.begin()) {
//        delete(*BondRunner);
//      }
      mol->AddAtom(*AtomRunner);
    }
  }

  // 1. create the bond structure of the single molecule
  LOG(0, "STATUS: (Re-)constructing adjacency.");
  if (mol->getBondCount() == 0) {
    BondGraph *BG = World::getInstance().getBondGraph();
    molecule::atomVector Set = mol->getAtomSet();
    BG->CreateAdjacency(Set);
//    if (mol->getBondCount() == 0) {
//      World::getInstance().destroyMolecule(mol);
//      ELOG(1, "There are no bonds.");
//      return Action::failure;
//    }
  }

  // 2. scan for connected subgraphs
  LOG(0, "STATUS: Analysing adjacency graph.");
  MoleculeLeafClass *Subgraphs = NULL;      // list of subgraphs from DFS analysis
  std::deque<bond *> *BackEdgeStack = NULL;
  Subgraphs = mol->DepthFirstSearchAnalysis(BackEdgeStack);
  delete(BackEdgeStack);
  if ((Subgraphs == NULL) || (Subgraphs->next == NULL)) {
    //World::getInstance().destroyMolecule(mol);
    DoeLog(1) && (eLog()<< Verbose(1) << "There are no atoms." << endl);
    return Action::failure;
  }

  //int FragmentCounter = Subgraphs->next->Count();

  // TODO: When DepthFirstSearchAnalysis does not use AddCopyAtom() anymore, we don't need to delete all original atoms
  LOG(0, "STATUS: Creating molecules as connected subgraphs.");
  {
    {
      atom **ListOfAtoms = NULL;
      // 3a. re-create bond structure and insert molecules into general MoleculeListClass
      MoleculeLeafClass *MoleculeWalker = Subgraphs->next;
      while (MoleculeWalker->next != NULL) {
        ListOfAtoms = NULL;
        MoleculeWalker->FillBondStructureFromReference(mol, ListOfAtoms, true);  // we want to keep the created ListOfLocalAtoms
        molecules->insert(MoleculeWalker->Leaf);
        MoleculeWalker = MoleculeWalker->next;
      }
      molecules->insert(MoleculeWalker->Leaf);
      ListOfAtoms = NULL;
      MoleculeWalker->FillBondStructureFromReference(mol, ListOfAtoms, true);  // we want to keep the created ListOfLocalAtoms
    }

    // 3b. store map from new to old ids for 3d
    vector <atom *> allatoms = World::getInstance().getAllAtoms();
    AtomAtomList newtooldlist;
    for(vector<atom *>::iterator AtomRunner = allatoms.begin(); AtomRunner != allatoms.end(); ++AtomRunner)
      if ((*AtomRunner)->father != (*AtomRunner))
        newtooldlist.insert( std::pair<atomId_t, atomId_t> ((*AtomRunner)->getId(),(*AtomRunner)->father->getId()) );

    {
      // 3c. destroy the original molecule
      for (molecule::iterator AtomRunner = mol->begin(); !mol->empty(); AtomRunner = mol->begin())
        World::getInstance().destroyAtom(*AtomRunner);
      World::getInstance().destroyMolecule(mol);
    }

    {
      // 3d. convert to old Ids and correct fathers (AddCopyAtom sets father to original atom, which has been destroyed).
      vector <atom *> allatoms = World::getInstance().getAllAtoms();
      for(vector<atom *>::iterator AtomRunner = allatoms.begin(); AtomRunner != allatoms.end(); ++AtomRunner) {
        World::getInstance().changeAtomId((*AtomRunner)->getId(), newtooldlist[(*AtomRunner)->getId()]);
        (*AtomRunner)->father = *AtomRunner;
      }
    }
  }

  // 4. free Leafs
  LOG(0, "STATUS: Done.");
  MoleculeLeafClass *MolecularWalker = Subgraphs;
  while (MolecularWalker->next != NULL) {
    MolecularWalker->Leaf = NULL;
    MolecularWalker = MolecularWalker->next;
    delete(MolecularWalker->previous);
  }
  MolecularWalker->Leaf = NULL;
  delete(MolecularWalker);
  DoLog(1) && (Log() << Verbose(1) << "I scanned " << molecules->ListOfMolecules.size() << " molecules." << endl);

  return Action::state_ptr(UndoState);
}

Action::state_ptr FragmentationSubgraphDissectionAction::performUndo(Action::state_ptr _state) {
  FragmentationSubgraphDissectionState *state = assert_cast<FragmentationSubgraphDissectionState*>(_state.get());

  {
    // remove all present molecules
    MoleculeListClass *molecules = World::getInstance().getMolecules();
    vector<molecule *> allmolecules = World::getInstance().getAllMolecules();
    for (vector<molecule *>::iterator MolRunner = allmolecules.begin(); MolRunner != allmolecules.end(); ++MolRunner) {
      molecules->erase(*MolRunner);
      World::getInstance().destroyMolecule(*MolRunner);
    }
  }

  {
    // construct the old state
    MoleculeListClass *molecules = World::getInstance().getMolecules();
    molecule *mol = NULL;
    for (MolAtomList::const_iterator iter = state->moleculelist.begin(); iter != state->moleculelist.end(); ++iter) {
      mol = World::getInstance().createMolecule();
      if (mol->getId() != (*iter).first)
        World::getInstance().changeMoleculeId(mol->getId(), (*iter).first);
      for (std::vector<atomId_t>::const_iterator atomiter = (*iter).second.begin(); atomiter != (*iter).second.end(); ++atomiter) {
        atom *Walker = World::getInstance().getAtom(AtomById(*atomiter));
        mol->AddAtom(Walker);
      }
      molecules->insert(mol);
    }
  }

  return Action::state_ptr(_state);
}

Action::state_ptr FragmentationSubgraphDissectionAction::performRedo(Action::state_ptr _state){
  return performCall();
}

bool FragmentationSubgraphDissectionAction::canUndo() {
  return true;
}

bool FragmentationSubgraphDissectionAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
