/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * ChronosUnitTest.cpp
 */


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include <limits>

#include "CodePatterns/Chronos.hpp"

#include "ChronosUnitTest.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( ChronosTest );


void dummy()
{
  Chronos::getInstance().startTiming(__func__);
  for (int i=0;i<10000000;++i)
    std::cout << "";
  Chronos::getInstance().endTiming(__func__);
}

void dummy_two()
{
  Chronos::getInstance().startTiming(__func__);
  for (int i=0;i<1000000;++i)
    std::cout << "";
  Chronos::getInstance().endTiming(__func__);
}

static size_t level = 0;

void dummyRecursion()
{
  if (level > 2)
    return;
  ++level;
  Chronos::getInstance().startTiming(__func__);
  for (int i=0;i<10;++i) {
    std::cout << "";
    dummyRecursion();
  }
  Chronos::getInstance().endTiming(__func__);
  --level;
}


void ChronosTest::setUp()
{
}

void ChronosTest::tearDown()
{
  Chronos::purgeInstance();
}

/**
 * UnitTest for Chronos()
 */
void ChronosTest::InstanceTest()
{
  // check that we get a Non-NULL pointer
  CPPUNIT_ASSERT(Chronos::getPointer());
}

/**
 * UnitTest for Chronos()
 */
void ChronosTest::dummyTest()
{
  double timings[5];
  // first dummy
  dummy();
  CPPUNIT_ASSERT( Chronos::getInstance().StartingTime.find(std::string("dummy"))
      == Chronos::getInstance().StartingTime.end() );
  CPPUNIT_ASSERT( Chronos::getInstance().AccountedTime.find(std::string("dummy"))
      != Chronos::getInstance().AccountedTime.end() );
  CPPUNIT_ASSERT_EQUAL( (size_t) 1, Chronos::getInstance().SumUpTotalFunctions() );
  timings[0] = Chronos::getInstance().AccountedTime[std::string("dummy")];
  std::cout << "Timing[0]: " << timings[0] << std::endl;
  CPPUNIT_ASSERT(timings[0] > 0.);

  // second call goes to same entry
  dummy();
  CPPUNIT_ASSERT_EQUAL( (size_t) 1, Chronos::getInstance().SumUpTotalFunctions() );
  timings[1] = Chronos::getInstance().AccountedTime[std::string("dummy")] - timings[0];
  std::cout << "Timing[1]: " << timings[1] << std::endl;
  CPPUNIT_ASSERT(timings[1] > 0.);

  // second dummy
  dummy_two();
  CPPUNIT_ASSERT( Chronos::getInstance().AccountedTime.find(std::string("dummy_two"))
      != Chronos::getInstance().AccountedTime.end() );
  CPPUNIT_ASSERT_EQUAL( (size_t) 2, Chronos::getInstance().SumUpTotalFunctions() );
  timings[2] = Chronos::getInstance().AccountedTime[std::string("dummy_two")];
  std::cout << "Timing[2]: " << timings[2] << std::endl;
  CPPUNIT_ASSERT(timings[2] > 0.);

  // recursive dummy
  dummyRecursion();
  CPPUNIT_ASSERT( Chronos::getInstance().RecursionMap.find(std::string("dummyRecursion"))
      != Chronos::getInstance().RecursionMap.end() );
  CPPUNIT_ASSERT( Chronos::getInstance().AccountedTime.find(std::string("dummyRecursion"))
      != Chronos::getInstance().AccountedTime.end() );
  CPPUNIT_ASSERT_EQUAL( (size_t) 3, Chronos::getInstance().SumUpTotalFunctions() );
  timings[3] = Chronos::getInstance().AccountedTime[std::string("dummyRecursion")];
  std::cout << "Timing[3]: " << timings[3] << std::endl;
  CPPUNIT_ASSERT(timings[3] > 0.);

  // "inline" dummy
  {
    Chronos::getInstance().startTiming("dummy_three");
    for (int i=0;i<1000000;++i)
      std::cout << "";
    Chronos::getInstance().endTiming("dummy_three");
  }
  CPPUNIT_ASSERT( Chronos::getInstance().AccountedTime.find(std::string("dummy_three"))
      != Chronos::getInstance().AccountedTime.end() );
  CPPUNIT_ASSERT_EQUAL( (size_t) 4, Chronos::getInstance().SumUpTotalFunctions() );
  timings[4] = Chronos::getInstance().AccountedTime[std::string("dummy_three")];
  std::cout << "Timing[4]: " << timings[4] << std::endl;
  CPPUNIT_ASSERT(timings[4] > 0.);

  // check summing of times
  CPPUNIT_ASSERT( fabs(timings[0] + timings[1] + timings[2] + timings[3] + timings[4]- Chronos::getInstance().SumUpTotalTime()) < numeric_limits<double>::epsilon());

  std::cout << Chronos::getInstance() << std::endl;
}

/**
 * UnitTest for Chronos::calculateCorrectTimeDifference()
 */
void ChronosTest::calculateCorrectTimeDifferenceTest()
{
  Chronos::sec_ncsec_t first;
  Chronos::sec_ncsec_t second;
	{
		// 1.1 - 0.01
		first.first = 1;
		first.second = 1e8;
		second.first = 0;
		second.second = 1e7;
		double difference = Chronos::calculateCorrectTimeDifference(first, second);
    CPPUNIT_ASSERT( fabs(1.1-0.01 - difference) < std::numeric_limits<double>::epsilon()*1e2);
//		CPPUNIT_ASSERT_EQUAL( 1.1-0.01, difference);
	}
	{
		// 1.1 - 0.9
		first.first = 1;
		first.second = 1e8;
		second.first = 0;
		second.second = 9e8;
		double difference = Chronos::calculateCorrectTimeDifference(first, second);
		//    CPPUNIT_ASSERT_EQUAL( 1.1-0.9, difference);
		CPPUNIT_ASSERT( fabs(1.1-0.9 - difference) < std::numeric_limits<double>::epsilon()*1e2);
	}
	{
		// 1000.1 - 0.9
		first.first = 1000;
		first.second = 1e8;
		second.first = 0;
		second.second = 9e8;
		double difference = Chronos::calculateCorrectTimeDifference(first, second);
//		CPPUNIT_ASSERT_EQUAL( 1000.1-0.9, difference);
    CPPUNIT_ASSERT( fabs(1000.1-0.9 - difference) < std::numeric_limits<double>::epsilon()*1e2);
	}
}
