/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * QtMainWindow.cpp
 *
 *  Created on: Jan 14, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "QtMainWindow.hpp"

#include<Qt/qapplication.h>
#include<Qt/qlabel.h>
#include<Qt/qstring.h>
#include<Qt/qmenubar.h>
#include<Qt/qsplitter.h>
#include<Qt/qtoolbar.h>
#include<Qt/qsettings.h>
#include<Qt/qmessagebox.h>
#include<Qt/qevent.h>

#include<iostream>
#include<map>

#include<boost/bind.hpp>

#include "Menu/Qt4/QtMenu.hpp"
#include "Views/Qt4/QtMoleculeList.hpp"
#include "Views/Qt4/QtElementList.hpp"
#include "Views/Qt4/QtInfoBox.hpp"
#include "Views/Qt4/QtStatusBar.hpp"
#include "Views/Qt4/QtToolBar.hpp"
#include "Views/Qt4/Qt3D/GLWorldView.hpp"

#include "CodePatterns/MemDebug.hpp"

#include "Atom/atom.hpp"
#include "molecule.hpp"
#include "CodePatterns/Verbose.hpp"
#include "Actions/Action.hpp"
#include "Actions/ActionRegistry.hpp"
#include "Parser/ChangeTracker.hpp"
#include "Parser/FormatParserStorage.hpp"
#include "Actions/WorldAction/OutputAction.hpp"
#include "Actions/WorldAction/OutputAsAction.hpp"

using namespace MoleCuilder;

QtMainWindow::QtMainWindow(QApplication *_theApp) :
    theApp(_theApp)
{
  QCoreApplication::setOrganizationName("ins");
  QCoreApplication::setOrganizationDomain("ins.uni-bonn.de");
  QCoreApplication::setApplicationName("MoleCuilder");
  QSplitter *splitter1 = new QSplitter (Qt::Horizontal, this );
  QSplitter *splitter2 = new QSplitter (Qt::Vertical, splitter1 );
  QTabWidget *worldTab = new QTabWidget(splitter2);

  moleculeList = new QtMoleculeList(worldTab);
  elementList = new QtElementList(worldTab);

  infoBox = new QtInfoBox();
  glWorldView = new GLWorldView(this);
//  glWorldView->setSizePolicy( QSizePolicy::Minimum,QSizePolicy::Minimum);
  glWorldView->setMinimumSize( QSize(640,480) );
//  glWorldView->setFocus();
  glWorldView->camera()->setEye( QVector3D(0,3,10));

  MainMenu = new QtMenu<QMenuBar>(menuBar(), "");
  MainMenu->init();

  toolbar = new QtToolBar(this);
  glWorldView->addToolBarActions(toolbar);
  addToolBar(toolbar);

  setCentralWidget(splitter1);
  splitter1->addWidget(splitter2);
  splitter1->addWidget(infoBox);
  splitter2->addWidget(glWorldView);
  splitter2->addWidget(worldTab);
  worldTab->addTab(moleculeList, "Molecules");
  worldTab->addTab(elementList, "All Elements");

  statusBar = new QtStatusBar(this);
  setStatusBar(statusBar);

  QSettings settings;
  settings.beginGroup("MainWindow");
  resize(settings.value("size", QSize(400, 400)).toSize());
  move(settings.value("position", QPoint(200, 200)).toPoint());
  if (settings.value("maximized", false).toBool())
    showMaximized();
  settings.endGroup();

  connect(glWorldView,SIGNAL(hoverChanged(const atom*)), infoBox,SLOT(atomHover(const atom*)));
}

QtMainWindow::~QtMainWindow()
{
  QSettings settings;
  settings.beginGroup("MainWindow");
  settings.setValue("size", size());
  settings.setValue("position", pos());
  settings.setValue("maximized", isMaximized());
  settings.endGroup();

  menuBar()->clear();
  delete MainMenu;
}

void QtMainWindow::display() {
  this->show();
  theApp->exec();
}

void QtMainWindow::closeEvent(QCloseEvent *event)
{
  if (ChangeTracker::getInstance().hasChanged()){
    int ret = QMessageBox::question(this, tr("MoleCuilder"),
                                    tr("The world has been modified.\n"
                                       "Do you want to save your changes?"),
                                    QMessageBox::Save | QMessageBox::Discard | QMessageBox::Cancel,
                                    QMessageBox::Save);
    switch(ret){
      case QMessageBox::Cancel:
        event->ignore();
        return;
      case QMessageBox::Save:
        if (FormatParserStorage::getInstance().isAbleToSave())
          MoleCuilder::WorldOutput();
        else{
          Action* action = ActionRegistry::getInstance().getActionByName("output-as");
          try{
            action->call();
          }catch(...){
            ELOG(1, "Action failed or cancelled");
          }
        }
        break;
    }
  }
  QMainWindow::closeEvent(event);
}

