/*
 * QtObservedInstanceBoard.hpp
 *
 *  Created on: Oct 17, 2015
 *      Author: heber
 */


#ifndef QTOBSERVEDINSTANCEBOARD_HPP_
#define QTOBSERVEDINSTANCEBOARD_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <QtGui/QWidget>

#include "UIElements/Qt4/InstanceBoard/QtObservedAtom.hpp"
#include "UIElements/Qt4/InstanceBoard/QtObservedMolecule.hpp"

#include <map>
#include <boost/function.hpp>

#include "CodePatterns/Observer/Observer.hpp"

#include "UIElements/Qt4/InstanceBoard/ObservedValue_types.hpp"
#include "UIElements/Qt4/InstanceBoard/ObservedValuesContainer.hpp"
#include "types.hpp"

class GLWorldScene;
class GLMoleculeObject_bond;
class GLMoleculeObject_molecule;

/** The QtObservedInstanceBoard is the central class for communicating instance
 * creation and destruction from MoleCuilder into the QtGui realm. It should be
 * thought of as an interface to allow for safe multi-threaded computing.
 *
 * The idea is as follows: As soon as a molecule is instantiated, all the QtGui
 * needs to access the instance (for displaying it) is wrapped up in a
 * ObservedValue. This ObservedValue separates the lifetime of the molecule object
 * from the information contained therein and thus makes the visual representation
 * independent of the life time. The Observer/Observable signal from the World,
 * i.e. World::MoleculeInserted, is caught (directly within the same thread) by
 * the QtObservedInstanceBoard. Here, we instantiate all ObservedValue's needed
 * for this specific molecule and store them in an internal map. Next, we emit
 * a Qt signal informing the QtGui part about the new molecule.
 * At some later stage, the QtGui will (probably in a different thread)
 * instantiate a GLMoleculeObject_molecule as a visual representation of the
 * molecule. It requests the ObservedValues from the QtObservedInstanceBoard
 * and uses these directly.
 * The QtObservedInstanceBoard also records all subjectKilled() signals from
 * each ObservedValue. Additionally, each class using the ObservedValues
 * additionally informs the QtObservedInstanceBoard when subjectKilled() was
 * called. If subjectKilled() for each ObservedValue of a molecule and from the
 * visual representation have been received, a removal Qt signal is emitted.
 *
 * The same holds for the atom
 */
class QtObservedInstanceBoard : public QWidget, public Observer
{
  Q_OBJECT

public:
  /** Cstor of QtObservedInstanceBoard.
   *
   * \param _parent Qt parent to automatically destroy when parent is destroyed
   */
  QtObservedInstanceBoard(QWidget * _parent=0);

  /** Dstor of QtObservedInstanceBoard.
   *
   */
  ~QtObservedInstanceBoard();

  // Observer functions
  void update(Observable *publisher);
  void subjectKilled(Observable *publisher);
  void recieveNotification(Observable *publisher, Notification_ptr notification);

  QtObservedAtom::ptr getObservedAtom(const atomId_t _id);
  QtObservedMolecule::ptr getObservedMolecule(const moleculeId_t _id);
  void markObservedAtomAsConnected(const atomId_t _id);
  void markObservedAtomAsDisconnected(const atomId_t _id);
  void markObservedAtomForErase(const atomId_t _id);
  void markObservedMoleculeAsConnected(const moleculeId_t _id);
  void markObservedMoleculeAsDisconnected(const moleculeId_t _id);
  void markObservedMoleculeForErase(const moleculeId_t _id);

signals:
  void atomInserted(QtObservedAtom::ptr _atom);
  void atomRemoved(const atomId_t _atomid);
  void atomIndexChanged(const atomId_t _oldid, const atomId_t _newid);
  void moleculeInserted(QtObservedMolecule::ptr _mol);
  void moleculeRemoved(const moleculeId_t _molid);
  void moleculeIndexChanged(const moleculeId_t _oldid, const moleculeId_t _newid);

private:
  friend class GLWorldScene;
  friend class GLMoleculeObject_bond;
  friend class QtObservedAtom;
  friend class QtObservedMolecule;

  //!> indicating whether we are still signedOn to World or not
  bool WorldSignedOn;

  typedef std::set<Observable *> SignedOn_t;
  //!> map indicating to which atom we are currently signed on
  SignedOn_t AtomSignedOn;
  //!> map indicating to which molecule we are currently signed on
  SignedOn_t MoleculeSignedOn;

  /** Counts how many atom's ObservedValues got subjectKilled.
   *
   * This is used to give removal signal only when each and every
   * ObservedValue (and the instance itself) has been subjectKilled by the
   * monitored Observable. Only then can we safely remove the instance.
   *
   * \param _atomid id of the atom
   */
  void atomcountsubjectKilled(const atomId_t _atomid);

  /** Counts how many molecule's ObservedValues got subjectKilled.
   *
   * This is used to give removal signal only when each and every
   * ObservedValue (and the instance itself) has been subjectKilled by the
   * monitored Observable. Only then can we safely remove the instance.
   *
   * \param _molid id of the molecule
   */
  void moleculecountsubjectKilled(const moleculeId_t _molid);

  //!> container with all ObservedValues for each atom, associated by id
  ObservedValuesContainer<QtObservedAtom, atomId_t> atomObservedValues;
  //!> container with all ObservedValues for each molecule, associated by id
  ObservedValuesContainer<QtObservedMolecule, moleculeId_t> moleculeObservedValues;
};

#endif /* QTOBSERVEDINSTANCEBOARD_HPP_ */
