/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2014 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * TranslateToOriginAction.cpp
 *
 *  Created on: Aug 14, 2014
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"
#include "Atom/atom.hpp"
#include "Atom/AtomSet.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "World.hpp"

#include <iostream>
#include <fstream>
#include <string>

#include "Actions/AtomAction/TranslateToOriginAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "TranslateToOriginAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
ActionState::ptr AtomTranslateToOriginAction::performCall() {

  if (World::getInstance().beginAtomSelection() == World::getInstance().endAtomSelection()) {
    STATUS("No atoms selected.");
    return Action::failure;
  }

  // calculate center of selected atoms
  Vector Translation;
  size_t No = 0;
  for (World::AtomSelectionIterator iter = World::getInstance().beginAtomSelection();
      iter != World::getInstance().endAtomSelection();
      ++iter, ++No) {
    Translation += (iter->second)->getPosition();
  }
  ASSERT( No != 0, "AtomTranslateToOriginAction::performCall() - No is nonetheless zero?");
  Translation *= -1./(double)No;

  World::AtomComposite atoms = World::getInstance().getAllAtoms();
  atoms.translate(Translation);
  STATUS("Atoms translated by "+toString(Translation));

  return ActionState::ptr(new AtomTranslateToOriginState(Translation, params));
}

ActionState::ptr AtomTranslateToOriginAction::performUndo(ActionState::ptr _state) {
  AtomTranslateToOriginState *state = assert_cast<AtomTranslateToOriginState*>(_state.get());

  World::AtomComposite atoms = World::getInstance().getAllAtoms();
  atoms.translate(-1.*state->Translation);
  STATUS("Undid translation by "+toString(state->Translation));

  return ActionState::ptr(_state);
}

ActionState::ptr AtomTranslateToOriginAction::performRedo(ActionState::ptr _state){
  AtomTranslateToOriginState *state = assert_cast<AtomTranslateToOriginState*>(_state.get());

  World::AtomComposite atoms = World::getInstance().getAllAtoms();
  atoms.translate(state->Translation);
  STATUS("Redid translation by "+toString(state->Translation));

  return ActionState::ptr(_state);
}

bool AtomTranslateToOriginAction::canUndo() {
  return true;
}

bool AtomTranslateToOriginAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
