/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * Copyright (C)  2013 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * BondInfo.cpp
 *
 *  Created on: Apr 5, 2012
 *      Author: heber
 */


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "BondInfo.hpp"

#include <algorithm>
#include <boost/bind.hpp>
#include <boost/function.hpp>

#include "Atom/atom.hpp"
#include "Bond/bond.hpp"
#include "CodePatterns/Log.hpp"
#include "Descriptors/AtomIdDescriptor.hpp"
#include "World.hpp"
#include "WorldTime.hpp"

BondInfo::BondInfo(const bond::ptr _bond) :
  leftid(_bond->leftatom->getId()),
  rightid(_bond->rightatom->getId()),
  degree(_bond->getDegree())
{}

bool BondInfo::RecreateBond() const
{
  atom * const leftatom = World::getInstance().getAtom(AtomById(leftid));
  atom * const rightatom = World::getInstance().getAtom(AtomById(rightid));
  if ((leftatom == NULL) || (rightatom == NULL)) {
    ELOG(1, "Either of the two atoms " << leftid << " or "
        << rightid << " does not exists in the world.");
    return false;
  }
  const BondList &leftbonds = leftatom->getListOfBonds();
  boost::function< bool (const bond::ptr )> predicate =
      boost::bind(static_cast<bool (bond::*)(const int) const>(&bond::Contains), _1, rightid);
  if (std::find_if(leftbonds.begin(), leftbonds.end(), predicate) != leftbonds.end())
    return false;

  const BondList &rightbonds = rightatom->getListOfBonds();
  if (std::find_if(rightbonds.begin(), rightbonds.end(), predicate) != rightbonds.end())
    return false;

  // create bond
  LOG(3, "DEBUG: Re-creating bond " << leftid << "<->" << rightid << ".");
  bond::ptr const _bond = leftatom->addBond(WorldTime::getTime(), rightatom);
  _bond->setDegree(degree);
  return true;
}
