/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * PeriodentafelUnitTest.cpp
 *
 *  Created on: May 18, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include <sstream>
#include <iostream>

#include "CodePatterns/Assert.hpp"

#include "Element/element.hpp"
#include "Element/elements_db.hpp"
#include "Element/periodentafel.hpp"

#include "PeriodentafelUnitTest.hpp"

// include headers that implement a archive in simple text format
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( periodentafelTest );


void periodentafelTest::setUp()
{
  // make sure world is present
  tafel = new periodentafel();
  CPPUNIT_ASSERT(tafel != NULL && "could not obtain periodentafel");
};


void periodentafelTest::tearDown()
{
  delete tafel;
};

/** UnitTest for AddRemoveTest().
 */
void periodentafelTest::AddRemoveTest()
{
  // copy the element
  const element * ElementPtr = tafel->FindElement(3);
  CPPUNIT_ASSERT( ElementPtr != NULL && "could not find element lithium");
  element *Elemental = new element(*ElementPtr);
  // remove element
  tafel->RemoveElement((element * const)ElementPtr);
  ElementPtr = tafel->FindElement(3);
  CPPUNIT_ASSERT_EQUAL( (const element *)NULL, ElementPtr );
  // add element again
  tafel->AddElement((element * const)Elemental);
  ElementPtr = tafel->FindElement(3);
  CPPUNIT_ASSERT( ElementPtr != NULL && "could not find re-added element lithium");
};

/** UnitTest for LoadStoreTest().
 * TODO: periodentafelTest::LoadStoreTest() not implemented yet
 */
void periodentafelTest::LoadStoreTest()
{
  // reload all databases
  stringstream elementsDBstream(elementsDB,ios_base::in);
  stringstream ElectronegativityDBstream(ElectronegativitiesDB,ios_base::in);
  stringstream valenceDBstream(valenceDB,ios_base::in);
  stringstream orbitalsDBstream(orbitalsDB,ios_base::in);
  stringstream HbondangleDBstream(HbondangleDB,ios_base::in);
  stringstream HbonddistanceDBstream(HbonddistanceDB,ios_base::in);
  stringstream ColorDBstream(ColorDB,ios_base::in);
  CPPUNIT_ASSERT(tafel->LoadElementsDatabase(elementsDBstream) && "General element initialization failed");
  CPPUNIT_ASSERT(tafel->LoadElectronegativityDatabase(ElectronegativityDBstream) && "Electronegativity entry of element initialization failed");
  CPPUNIT_ASSERT(tafel->LoadValenceDatabase(valenceDBstream) && "Valence entry of element initialization failed");
  CPPUNIT_ASSERT(tafel->LoadOrbitalsDatabase(orbitalsDBstream) && "Orbitals entry of element initialization failed");
  CPPUNIT_ASSERT(tafel->LoadHBondAngleDatabase(HbondangleDBstream) && "HBond angle entry of element initialization failed");
  CPPUNIT_ASSERT(tafel->LoadHBondLengthsDatabase(HbonddistanceDBstream) && "HBond distance entry of element initialization failed");
  CPPUNIT_ASSERT(tafel->LoadColorDatabase(ColorDBstream) && "color entry of element initialization failed");
  // check presence of elements
  atomicNumber_t Z = 1;
  for(periodentafel::const_iterator ElementRunner = tafel->begin(); ElementRunner != tafel->end(); ++ElementRunner)
    CPPUNIT_ASSERT( (ElementRunner->second->getNumber() == Z++) && "element is missing in sequence");
};

/** UnitTest for serialization
 *
 */
void periodentafelTest::ComparisonTest()
{
  periodentafel *newtafel = new periodentafel();

  CPPUNIT_ASSERT (*newtafel == *tafel );

  element *Elemental = new element();
  newtafel->AddElement((element * const)Elemental);

  CPPUNIT_ASSERT (*newtafel != *tafel );

  delete newtafel;
}

std::string gatherUndoInformation(const periodentafel *tafel)
{
  // create undo state
  std::stringstream undostream;
  boost::archive::text_oarchive oa(undostream);
  oa << tafel;
  return undostream.str();
}

/** UnitTest for serialization
 *
 */
void periodentafelTest::SerializeTest()
{
  // write element to stream
  std::stringstream stream(gatherUndoInformation(tafel));
//  boost::archive::text_oarchive oa(stream);
//  oa << World::getInstance().getPeriode();
//  //oa << tafel;
//
//  //std::cout << "Contents of archive is " << stream.str() << std::endl;
//
//  // create and open an archive for input
  boost::archive::text_iarchive ia(stream);
  // read class state from archive
  periodentafel * newtafel;

  ia >> newtafel;

  CPPUNIT_ASSERT (*tafel == *newtafel);
}
