/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * QtStatusBar.cpp
 *
 *  Created on: Feb 17, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sstream>

#include <QtGui/QLabel>
#include <QtGui/QBoxLayout>
#include <QtGui/QProgressBar>

#include "QtStatusBar.hpp"

#include "CodePatterns/MemDebug.hpp"

#include "World.hpp"
#include "Helpers/helpers.hpp"
#include "Actions/Process.hpp"

#define PLURAL_S(v) (((v)==1)?"":"s")


QtStatusBar::QtStatusBar(QWidget *_parent) :
  QStatusBar(_parent),
  Observer("QtStatusBar"),
  atomCount(World::getInstance().numAtoms()),
  moleculeCount(World::getInstance().numMolecules()),
  parent(_parent)
{
  World::getInstance().signOn(this);
  Process::AddObserver(this);
  statusLabel = new QLabel(this);
  statusLabel->setFrameStyle(QFrame::NoFrame | QFrame::Plain);
  addPermanentWidget(statusLabel);
  redrawStatus();
}

QtStatusBar::~QtStatusBar()
{
  Process::RemoveObserver(this);
  World::getInstance().signOff(this);
}

void QtStatusBar::update(Observable *subject){
  if (subject==World::getPointer()){
    atomCount = World::getInstance().numAtoms();
    moleculeCount = World::getInstance().numMolecules();
    redrawStatus();
  }
  else {
    // we probably have some process
    Process *proc;
    if((proc=dynamic_cast<Process*>(subject))){
      redrawProcess(proc);
    }
  }
}

void QtStatusBar::subjectKilled(Observable *subject){
  // Processes don't notify when they are killed
  atomCount = World::getInstance().numAtoms();
  moleculeCount = World::getInstance().numMolecules();
  World::getInstance().signOn(this);
  redrawStatus();
}

void QtStatusBar::redrawStatus(){
  stringstream sstr;
  sstr << "You have " << atomCount << " atom" << PLURAL_S(atomCount)
       <<" in " << moleculeCount << " molecule" << PLURAL_S(moleculeCount);
  statusLabel->setText(QString(sstr.str().c_str()));
}

void QtStatusBar::redrawProcess(Process *proc){
  progressIndicator *ind=0;
  // see what we have to do with the process
  if(proc->doesStart()){
    ind = new progressIndicator(proc->getName());
    ind->bar->setMaximum(proc->getMaxSteps());
    progressBars.insert(pair<Process*,progressIndicator*>(proc,ind));
  }
  else {
    ind = progressBars[proc];
  }
  if(activeProcess!=proc){
    addWidget(ind->container);
    activeProcess = proc;
  }
  ind->bar->setValue(proc->getCurrStep());
  parent->repaint();
  if(proc->doesStop()){
    removeWidget(ind->container);
    activeProcess = 0;
    progressBars.erase(proc);
    delete ind;
  }
}



QtStatusBar::progressIndicator::progressIndicator(string name){
  stringstream sstr;
  sstr << "Busy (" << name << ")";
  container = new QWidget();
  layout = new QHBoxLayout(container);
  label = new QLabel(QString(sstr.str().c_str()));
  bar = new QProgressBar();

  layout->addWidget(label);
  layout->addWidget(bar);
  container->setLayout(layout);
}

QtStatusBar::progressIndicator::~progressIndicator(){
  delete container;
}
