/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * VMGFragmentController.cpp
 *
 *  Created on: Aug 27, 2012
 *      Author: heber
 */


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// boost asio needs specific operator new
#include <boost/asio.hpp>

#include "CodePatterns/MemDebug.hpp"

#include "VMGFragmentController.hpp"

#include "Atom/atom.hpp"
#include "Element/element.hpp"
#include "Helpers/defs.hpp"
#include "Jobs/VMGJob.hpp"
#include "molecule.hpp"
#include "World.hpp"

bool VMGFragmentController::createLongRangeJobs(
    const std::map<JobId_t, MPQCData> &fragmentData,
    const std::vector<SamplingGrid> &full_sampled_grid,
    const size_t near_field_cells,
    const size_t interpolation_degree)
{
  std::vector<FragmentJob::ptr> jobs;
  // add one job for each fragment as the short-range correction which we need
  // to subtract from the obtained full potential to get the long-range part only
  for (std::map<JobId_t, MPQCData>::const_iterator iter = fragmentData.begin();
      iter != fragmentData.end(); ++iter) {
    const JobId_t next_id = getAvailableId();
    const MPQCData &data = iter->second;
    LOG(1, "INFO: Creating VMGJob with " << data.sampled_grid
        << " gridpoints and " << data.charges.size() << " particle charges.");
    FragmentJob::ptr testJob(
        new VMGJob(next_id, data.sampled_grid, data.positions, data.charges, near_field_cells, interpolation_degree) );
    jobs.push_back(testJob);
  }

  // prepare positions and charges of full system
  std::vector< std::vector<double> > positions;
  std::vector<double> charges;
  {
    const World::AtomComposite &atoms = World::getInstance().getAllAtoms();
    positions.reserve(atoms.size());
    charges.reserve(atoms.size());
    std::vector<double> position(3, 0.);
    for (World::AtomComposite::const_iterator iter = atoms.begin();
        iter != atoms.end(); ++iter) {
      const Vector &pos = (*iter)->getPosition();
      // convert positions to atomic length units
      for (size_t i=0;i<3;++i) position[i] = pos[i]/AtomicLengthToAngstroem;
      positions.push_back(position);
      charges.push_back((double)((*iter)->getElement().getAtomicNumber()));
    }
  }
  for(std::vector<SamplingGrid>::const_iterator iter = full_sampled_grid.begin();
      iter != full_sampled_grid.end();
      ++iter) {
    const JobId_t next_id = getAvailableId();
    LOG(1, "INFO: Creating full VMGJob with " << *iter
        << " gridpoints and " << charges.size() << " particle charges.");
    FragmentJob::ptr testJob(
        new VMGJob(next_id, *iter, positions, charges, near_field_cells, interpolation_degree) );
    jobs.push_back(testJob);
  }

  // then send jobs to controller
  addJobs(jobs);
  sendJobs(host, port);
  RunService("Adding VMGJobs");

  return true;
}
