/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * ThermoStatContainer.cpp
 *
 *  Created on: 12.06.2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include <cstring>

#include "CodePatterns/Log.hpp"
#include "Thermostats/ThermoStatContainer.hpp"
#include "Parser/ConfigFileBuffer.hpp"

#include <Thermostats/Berendsen.hpp>
#include <Thermostats/GaussianThermostat.hpp>
#include <Thermostats/Langevin.hpp>
#include <Thermostats/NoseHoover.hpp>
#include <Thermostats/NoThermostat.hpp>
#include <Thermostats/Woodcock.hpp>

/** Constructor for class ThermoStatContainer.
 *
 */
ThermoStatContainer::ThermoStatContainer() :
  activeThermostat(0),
  TargetTemp(0.00095004455)
{
  ThermostatTraits<Thermostat> *BerendsenTrait = new ThermostatTraits<Berendsen>();
  availThermostats[BerendsenTrait->getName()] = BerendsenTrait;
  ThermostatTraits<Thermostat> *GaussianTrait = new ThermostatTraits<GaussianThermostat>();
  availThermostats[GaussianTrait->getName()] = GaussianTrait;
  ThermostatTraits<Thermostat> *LangevinTrait = new ThermostatTraits<Langevin>();
  availThermostats[LangevinTrait->getName()] = LangevinTrait;
  ThermostatTraits<Thermostat> *NoseHooverTrait = new ThermostatTraits<NoseHoover>();
  availThermostats[NoseHooverTrait->getName()] = NoseHooverTrait;
  ThermostatTraits<Thermostat> *NoThermostatTrait = new ThermostatTraits<NoThermostat>();
  availThermostats[NoThermostatTrait->getName()] = NoThermostatTrait;
  ThermostatTraits<Thermostat> *WoodcockTrait = new ThermostatTraits<Woodcock>();
  availThermostats[WoodcockTrait->getName()] = WoodcockTrait;

  // for debugging: list all thermostats
//  cout << "List of known thermostats: ";
//  for(traitsMap::iterator iter = availThermostats.begin();iter!=availThermostats.end();++iter){
//    cout << iter->first << " ";
//  }
//  cout << endl;

  ASSERT(availThermostats.size()==6,"Not all implemented thermostats referenced!\nDid you check the names in the traits?");
  activeThermostat=new Berendsen();
  activeThermostat->addToContainer(this);
}

ThermostatTraits<Thermostat> *ThermoStatContainer::getTraitByName(const std::string name){
  if(!availThermostats.count(name))
    return 0;
  return availThermostats[name];
}

Thermostat *ThermoStatContainer::makeByName(const std::string name,class ConfigFileBuffer * const fb){
  ThermostatTraits<Thermostat>* trait = getTraitByName(name);
  if(trait){
    Thermostat *res = trait->make(fb);
    res->addToContainer(this);
    return res;
  }
  else{
    return 0;
  }
}

void ThermoStatContainer::makeActive(const std::string name,class ConfigFileBuffer * const fb){
  Thermostat* newThermostat = makeByName(name,fb);
  if(newThermostat){
    if(activeThermostat){
      delete activeThermostat;
    }
    activeThermostat = newThermostat;
  }
}

void ThermoStatContainer::chooseNone(){
  if(activeThermostat){
    delete activeThermostat;
  }
  activeThermostat = new NoThermostat();
}

/** Destructor for Class ThermoStatContainer.
 *
 */
ThermoStatContainer::~ThermoStatContainer()
{
  if(activeThermostat){
    delete activeThermostat;
  }
  for(traitsMap::iterator iter= availThermostats.begin();iter!=availThermostats.end();++iter){
    delete (iter->second);
  }
}


