/*
 * QtObservedBond.hpp
 *
 *  Created on: Mar 03, 2016
 *      Author: heber
 */


#ifndef QTOBSERVEDBOND_HPP_
#define QTOBSERVEDBOND_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <QtGui/QWidget>

#include <boost/function.hpp>
#include <boost/shared_ptr.hpp>

#include "CodePatterns/Observer/Observable.hpp"
#include "CodePatterns/Observer/Observer.hpp"

#include "LinearAlgebra/Vector.hpp"

#include "Bond/bond.hpp"
#include "UIElements/Qt4/InstanceBoard/ObservedValue_types.hpp"
#include "UIElements/Qt4/InstanceBoard/ObservedValuesContainer.hpp"
#include "UIElements/Qt4/InstanceBoard/QtObservedAtom.hpp"
#include "types.hpp"

class atom;
class molecule;
class QtObservedInstanceBoard;
class QtObservedMolecule;

/** This instance is the ObservedValue representation of a World's bond.
 *
 * Due to the signal/slot mechanism and its delays, lifetime of objects in the
 * World and their QtGui representation cannot directly be related (without
 * slowing down Actions just for having the representation up to speed).
 * Hence, the required information for displaying and representing these
 * objects must be contained in an extra instance.
 *
 * This is the instance for information about a particular bond.
 *
 * Essentially, this is the interface between molecuilder's World (and a
 * particular bond) and the QtGui part of the code.
 */
class QtObservedBond : public QWidget, public Observer
{
  Q_OBJECT

public:
  typedef std::pair<atomId_t, atomId_t> bondId_t;

  //!> typedef for instance wrapped in shared ptr
  typedef boost::shared_ptr<QtObservedBond> ptr;

  //!> typedef for instance wrapped in weak shared ptr
  typedef boost::weak_ptr<QtObservedBond> weak_ptr;

private:
  //!> ObservedValuesContainer needs to access private cstor and dstor
  friend class ObservedValuesContainer<QtObservedBond, ObservedValue_Index_t>;
  //!> QtObservedInstanceBoard needs to access private cstor and dstor
  friend class QtObservedInstanceBoard;

  /** Cstor of QtObservedBond.
   *
   * \param _id id of observed bond
   * \param _bond ref to observed bond
   * \param _leftatom bond's observed left atom for position, element, ...
   * \param _rightatom bond's observed right atom for position, element, ...
   * \param _board ref to InstanceBoard for callbacks on occasion of subjectKilled()
   * \param _parent Qt parent to automatically destroy when parent is destroyed
   */
  QtObservedBond(
      const bondId_t _id,
      const bond::ptr _bond,
      const QtObservedAtom::ptr &_leftatom,
      const QtObservedAtom::ptr &_rightatom,
      QtObservedInstanceBoard &_board,
      QWidget * _parent=0);

public:

  /** Dstor of QtObservedBond.
   *
   */
  ~QtObservedBond();

  // Observer functions
  void update(Observable *publisher);
  void subjectKilled(Observable *publisher);
  void recieveNotification(Observable *publisher, Notification_ptr notification);

  /** Getter for a permanent and unique index of this instance.
   *
   * \note ALWAYS use this index if you need to store and identifier to this
   * instance which you might need to retrieve at some later date.
   *
   * \warning DO NOT STORE the QtObserved...:ptr directly. This will possibly
   * prevent their eventual destruction. Only store the ObservedValue_Index_t
   * as means to obtain the ptr from the QtObservedInstanceBoard.
   *
   * \return returns a unique and permanent index that can be used to retrieve this
   * instance from the QtObservedInstanceBoard as it must not be stored.
   */
  ObservedValue_Index_t getIndex() const;

  /** Getter to bond's index pair contained in \a ObservedValues.
   *
   * \return bond's index pair
   */
  const bondId_t getBondIndex() const;

  /** Getter to bond's degree contained in \a ObservedValues.
   *
   * \return bond's degree
   */
  const int& getBondDegree() const;

  /** Getter to the observed state of the left atom of this bond.
   *
   * \return const ref to left atom's observed state.
   */
  const QtObservedAtom* const getLeftAtom() const
  { return leftatom.get(); }

  /** Getter to the observed state of the right atom of this bond.
   *
   * \return const ref to right atom's observed state.
   */
  const QtObservedAtom* const getRightAtom() const
  { return rightatom.get(); }

  /** Getter to bond's left atom index contained in \a ObservedValues.
   *
   * \return bond's left atom index
   */
  const atomId_t& getLeftAtomIndex() const;

  /** Getter to bond's left atom element contained in \a ObservedValues.
   *
   * \return bond's left atom element
   */
  const atomicNumber_t& getLeftAtomElement() const;

  /** Getter to left atoms's molecule index contained in \a ObservedValues.
   *
   * \return left atom's molecule index
   */
  const moleculeId_t getLeftMoleculeIndex() const;

  /** Getter to bond's right atom position contained in \a ObservedValues.
   *
   * \return bond's left atom position
   */
  const Vector& getLeftAtomPosition() const;

  /** Getter to bond's right atom index contained in \a ObservedValues.
   *
   * \return bond's right atom index
   */
  const atomId_t& getRightAtomIndex() const;

  /** Getter to bond's right atom element contained in \a ObservedValues.
   *
   * \return bond's right atom element
   */
  const atomicNumber_t& getRightAtomElement() const;

  /** Getter to bond's right atom position contained in \a ObservedValues.
   *
   * \return bond's right atom position
   */
  const Vector& getRightAtomPosition() const;

  /** Getter to right atom's molecule index contained in \a ObservedValues.
   *
   * \return right atom's molecule index
   */
  const moleculeId_t getRightMoleculeIndex() const;

  //!> typedef for internal observable counter maps
  typedef std::map<Observable * const, unsigned int> ObservableCount_t;

signals:

  void indexChanged();
  void degreeChanged();
  void leftAtomIndexChanged();
  void leftAtomElementChanged();
  void leftAtomPositionChanged();
  void leftmoleculeChanged();
  void rightAtomIndexChanged();
  void rightAtomElementChanged();
  void rightAtomPositionChanged();
  void rightmoleculeChanged();
  void bondRemoved();

//private slots:

private:
  void activateObserver();
  void deactivateObserver();

  static const atom * const getAtomConst(const atomId_t _id);
  static atom * const getAtom(const atomId_t _id);

private:
  static int updateDegree(const bond &_bond);

  //!> enumeration of observed values to match with entries in ObservedValues
  enum ObservedTypes {
    //!> contains the current bond degree
    BondDegree,
    //!> gives the size of the enumeration
    MAX_ObservedTypes
  };

  /** Initializes all \a ObservedValues entries.
   *
   * \param _ObservedValues vector of ObservedValue to be filled
   * \param _id bond id
   * \param _bondref reference to bond
   * \param _bondsubjectKilled ref to function to call on subjectKilled() from bond
   */
  void initObservedValues(
      ObservedValues_t &_ObservedValues,
      const bondId_t _id,
      const bond::ptr _bondref,
      const boost::function<void()> &_bondsubjectKilled);

  /** Destroys all \a ObservedValues entries.
   *
   * \param _ObservedValues vector of ObservedValue to be destroyed
   */
  static void destroyObservedValues(
      std::vector<boost::any> &_ObservedValues);

  /** Function is called by InstanceBoard to inform about its destruction.
   *
   * \note callbacks must not be used after this
   */
  void noteBoardIsGone()
  { BoardIsGone = true; }

  /** Counts the number of subject killed received from the observed values.
   *
   * \param _id id to check against ours
   * \param _counter counter to decrease
   */
  void countValuesSubjectKilled(ObservedValue_Index_t _id, unsigned int &_counter);

  //!> counts how many ObservedValues have already been subjectKilled() for a given observable
  mutable ObservableCount_t subjectKilledCount;

  /** Helper function to check that all subjectKilled have been received for both
   * this instance and all its internal observed values.
   *
   * \param _id id to check against ours
   */
  void checkForRemoval(ObservedValue_Index_t _id);

private:

  //!> list of channels when index needs to update
  static const Observable::channels_t BondDegreeChannels;

  //!> we get multiple subjectKilled(), count and call callback() only on last
  const ObservableCount_t AllsignedOnChannels;
  ObservableCount_t signedOffChannels;

  //!> the Observable we are signed on, also indicates whether we are sign on (not NULL)
  const Observable *bondowner;

private:

  /** Internal setter for the weak shared_ptr instance that we sometimes
   * need to convert the ref to this instance into an shared ptr instance that
   * is safe to hand around.
   *
   * \param _selfref ref to shared ptr instance that is internally stored
   */
  void setSelfRef(const weak_ptr &_selfref)
  { const_cast<weak_ptr &>(selfref) = _selfref; }

  //!> reference to this instance wrapped in a shared ptr for handing around
  const weak_ptr selfref;

public:

  /** Getter for this instance safely wrapped in a shared ptr instance for
   * handing arount.
   *
   * \return shared ptr of this instance
   */
  ptr getRef() const
  { return ptr(selfref); }

private:
  //!> contains still the old id after the index of the bond changed
  bondId_t oldbondId;

  //!> observed left atom of bond for position, element, ...
  const QtObservedAtom::ptr leftatom;

  //!> observed right atom of bond for position, element, ...
  const QtObservedAtom::ptr rightatom;

  //!> unique index among all observed instances
  const ObservedValue_Index_t index;

  //!> reference to InstanceBoard for callbacks on subjectKilled()
  QtObservedInstanceBoard & board;

  //!> is board still alive or not, impacts callbacks
  bool BoardIsGone;

  //!> internal reference to ObservedValues held by QtObservedInstanceBoard
  ObservedValues_t ObservedValues;
};


#endif /* QTOBSERVEDBOND_HPP_ */
