/** \file helpers.cpp
 *
 * Implementation of some auxiliary functions for memory dis-/allocation and so on
 */


#include "helpers.hpp"

/********************************************** helpful functions *********************************/


/** Asks for a double value and checks input
 * \param *text question
 */
double ask_value(const char *text)
{
	double test = 0.1439851348959832147598734598273456723948652983045928346598365;
	do {
		cout << Verbose(0) << text;
		cin >> test;
	} while (test == 0.1439851348959832147598734598273456723948652983045928346598365);
	return test;
};

/** Output of a debug message to stderr.
 * \param *P Problem at hand, points to ParallelSimulationData#me
 * \param output output string
 */
#ifdef HAVE_DEBUG
void debug_in(const char *output, const char *file, const int line) {
	if (output) fprintf(stderr,"DEBUG: in %s at line %i: %s\n", file, line, output);
}
#else
void debug_in(const char *output, const char *file, const int line) {}	// print nothing
#endif

/** Wrapper for allocating'ing a memory range with *output if it fails.
 * \param size number of chars to alloc for \a *buffer
 * \param *output message if malloc fails
 * \return pointer to memory range
 */
void * Malloc(size_t size, const char* output)
{
	void *buffer = NULL;
	buffer = (void *)malloc(size); // alloc
	if (buffer == NULL)
		cout << Verbose(0) << "Malloc failed - pointer is NULL: " << output << endl;
	return(buffer);
};

/** Wrapper for allocating'ing a memory range with *output if it fails.
 * \param size number of chars to alloc for \a *buffer
 * \param *output message if malloc fails
 * \return pointer to memory range
 */
void * Calloc(size_t size, const char* output)
{
	void *buffer = NULL;
	buffer = (void *)calloc(size, (size_t)0); // alloc
	if (buffer == NULL)
		cout << Verbose(0) << "Calloc failed - pointer is NULL: " << output << endl;
	return(buffer);
};

/** Wrapper for reallocating'ing a memory range with *output if it fails.
 * \param *OldPointer pointer to old memory range
 * \param size number of chars to alloc for \a *buffer
 * \param *output message if malloc fails
 * \return pointer to memory range
 */
void * ReAlloc(void * OldPointer, size_t size, const char* output)
{
  void *buffer = NULL;
  if (OldPointer == NULL)
    //cout << Verbose(0) << "ReAlloc impossible - old is NULL: " << output << endl;
    buffer = (void *)malloc(size); // malloc
  else
    buffer = (void *)realloc(OldPointer, size); // realloc
  if (buffer == NULL)
    cout << Verbose(0) << "ReAlloc failed - new is NULL: " << output << endl;
  return(buffer);
};

/** Wrapper for free'ing an allocated memory range with *output if it fails.
 * \param *buffer pointer to buffer to be free'd
 * \param *output message if free fails
 */
void Free(void ** buffer, const char* output)
{
	if (*buffer == NULL) {
		//cout << Verbose(5) << "Free not necesary: " << output << endl;
	} else {
		free(*buffer);
		*buffer = NULL;
	}
};

/** Malloc string array and set its length to the allocated length.
 * \param *output message if malloc fails
 * \param size number of chars to alloc for \a *buffer
 * \return pointer to string array
 */
char* MallocString(size_t size, const char* output)
{
	size_t i;
	char *buffer;
	buffer = (char *)malloc(sizeof(char) * (size+1)); // alloc
	if (buffer == NULL)
		cout << Verbose(0) << output << endl;
	for (i=size;i--;)	// reset
		buffer[i] = i % 2 == 0 ? 'p': 'c';
	buffer[size] = '\0'; // and set length marker on its end
	return(buffer);
}

/** modulo operator for doubles.
 * \param *b pointer to double
 * \param lower_bound lower bound
 * \param upper_bound upper bound
 */
void bound(double *b, double lower_bound, double upper_bound)
{
	double step = (upper_bound - lower_bound);
	while (*b >= upper_bound)
		*b -= step;
	while (*b < lower_bound)
		*b += step;
};

/** Flips two doubles.
 * \param *x pointer to first double
 * \param *y pointer to second double
 */
void flip(double *x, double *y)
{
	double tmp;
	tmp = *x;
	*x = *y;
	*y = tmp;
};

/** Returns the power of \a n with respect to \a base.
 * \param base basis
 * \param n power
 * \return \f$base^n\f$
 */
int pot(int base, int n)
{
	int res = 1;
	int j;
	for (j=n;j--;)
		res *= base;
	return res;
};

/** Returns a string with \a i prefixed with 0s to match order of total number of molecules in digits.
 * \param FragmentNumber total number of fragments to determine necessary number of digits
 * \param digits number to create with 0 prefixed
 * \return allocated(!) char array with number in digits, ten base.
 */
char *FixedDigitNumber(const int FragmentNumber, const int digits)
{
	char *returnstring;
	int number = FragmentNumber;
	int order = 0;
	while (number != 0) { // determine number of digits needed
		number = (int)floor(((double)number / 10.));
		order++;
		//cout << "Number is " << number << ", order is " << order << "." << endl;
	}
	// allocate string
	returnstring = (char *) Malloc(sizeof(char)*(order+2), "FixedDigitNumber: *returnstring");
	// terminate	and fill string array from end backward
	returnstring[order] = '\0';
	number = digits;
	for (int i=order;i--;) {
		returnstring[i] = '0' + (char)(number % 10);
		number = (int)floor(((double)number / 10.));
	}
	//cout << returnstring << endl;
	return returnstring;
};

/** Tests whether a given string contains a valid number or not.
 * \param *string
 * \return true - is a number, false - is not a valid number
 */
bool IsValidNumber( const char *string)
{
	int ptr = 0;
	if ((string[ptr] == '.') || (string[ptr] == '-')) // number may be negative or start with dot
		ptr++;
	if ((string[ptr] >= '0') && (string[ptr] <= '9'))
		return true;
	return false;
};


