/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/** \file FormatParserStorage.cpp
 *
 *  date: Jun, 22 2010
 *  author: heber
 *
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Helpers/MemDebug.hpp"

#include <iostream>
#include <fstream>

#include "Parser/FormatParserStorage.hpp"

#include "Parser/FormatParser.hpp"
#include "Parser/MpqcParser.hpp"
#include "Parser/PcpParser.hpp"
#include "Parser/TremoloParser.hpp"
#include "Parser/XyzParser.hpp"

#include "Helpers/Log.hpp"
#include "Helpers/Verbose.hpp"

#include "Helpers/Assert.hpp"

#include "Patterns/Singleton_impl.hpp"

/** Increment operator for the enumeration ParserTypes to allow loops.
 * \param &type value
 * \return value incremented by one
 */
ParserTypes &operator++(ParserTypes &type)
{
  return type = ParserTypes(type+1);
}

/** Constructor of class FormatParserStorage.
 */
FormatParserStorage::FormatParserStorage()
{
  ParserList.resize(ParserTypes_end, NULL);
  ParserStream.resize(ParserTypes_end, NULL);
  ParserPresent.resize(ParserTypes_end, false);
  ParserSuffix.resize(ParserTypes_end, "");

  ParserSuffix[mpqc] = "in";
  ParserSuffix[pcp] = "conf";
  ParserSuffix[tremolo] = "data";
  ParserSuffix[xyz] = "xyz";
}

/** Destructor of class FormatParserStorage.
 * Free all stored FormatParsers. 
 * Save on Exit.
 */
FormatParserStorage::~FormatParserStorage()
{
  for (ParserTypes iter = ParserTypes_begin; iter < ParserTypes_end; ++iter)
    if (ParserPresent[iter]) {
      if (ParserStream[iter]->is_open())
        ParserStream[iter]->close();
      delete ParserStream[iter];
      delete ParserList[iter];
    }
}

/** Sets the filename of all current parsers in storage to prefix.suffix.
 * \param &prefix prefix to use.
 */
void FormatParserStorage::SetOutputPrefixForAll(std::string &_prefix)
{
  prefix=_prefix;
};


void FormatParserStorage::SaveAll()
{
  std::string filename;
  for (ParserTypes iter = ParserTypes_begin; iter < ParserTypes_end; ++iter)
    if (ParserPresent[iter]) {
      filename = prefix;
      filename += ".";
      filename += ParserSuffix[iter];
      ParserStream[iter] = new std::ofstream(filename.c_str());
      ParserList[iter]->setOstream((std::ostream *)ParserStream[iter]);
    }
}


/** Adds an MpqcParser to the storage.
 */
void FormatParserStorage::addMpqc()
{
  if (!ParserPresent[mpqc]) {
    ParserList[mpqc] = dynamic_cast<FormatParser *>(new MpqcParser);
    ParserPresent[mpqc] = true;
  }
  else
    DoeLog(1) && (eLog() << Verbose(1) << "Parser mpqc is already present." << endl);
}

/** Adds an PcpParser to the storage.
 */
void FormatParserStorage::addPcp()
{
  if (!ParserPresent[pcp]) {
    ParserList[pcp] = new PcpParser();
    ParserPresent[pcp] = true;
  } else
    DoeLog(1) && (eLog() << Verbose(1) << "Parser pcp is already present." << endl);
}


/** Adds an TremoloParser to the storage.
 */
void FormatParserStorage::addTremolo()
{
  if (!ParserPresent[tremolo]) {
    ParserList[tremolo] = new TremoloParser();
    ParserPresent[tremolo] = true;
  } else
    DoeLog(1) && (eLog() << Verbose(1) << "Parser tremolo is already present." << endl);
}


/** Adds an XyzParser to the storage.
 */
void FormatParserStorage::addXyz()
{
  if (!ParserPresent[xyz]) {
    ParserList[xyz] = new XyzParser();
    ParserPresent[xyz] = true;
  } else
    DoeLog(1) && (eLog() << Verbose(1) << "Parser xyz is already present." << endl);
}

/** Parses an istream depending on its suffix
 * \param &input input stream
 * \param suffix
 * \return true - parsing ok, false - suffix unknown
 */
bool FormatParserStorage::get(std::istream &input, std::string suffix)
{
  if (suffix == ParserSuffix[mpqc]) {
    getMpqc().load(&input);
  } else if (suffix == ParserSuffix[pcp]) {
    getPcp().load(&input);
  } else if (suffix == ParserSuffix[tremolo]) {
    getTremolo().load(&input);
  } else if (suffix == ParserSuffix[xyz]) {
    getXyz().load(&input);
  } else {
    DoeLog(1) && (eLog() << Verbose(1) << "Unknown suffix to for FormatParserStorage::get()." << endl);
    return false;
  }
  return true;
}

/** Returns reference to the output MpqcParser, adds if not present.
 * \return reference to the output MpqcParser
 */
MpqcParser &FormatParserStorage::getMpqc()
{
  if (!ParserPresent[mpqc])
    addMpqc();
  return dynamic_cast<MpqcParser &>(*ParserList[mpqc]);
}

/** Returns reference to the output PcpParser, adds if not present.
 * \return reference to the output PcpParser
 */
PcpParser &FormatParserStorage::getPcp()
{
  if (!ParserPresent[pcp])
    addPcp();
  return dynamic_cast<PcpParser &>(*ParserList[pcp]);
}

/** Returns reference to the output TremoloParser, adds if not present.
 * \return reference to the output TremoloParser
 */
TremoloParser &FormatParserStorage::getTremolo()
{
  if (!ParserPresent[tremolo])
    addTremolo();
  return dynamic_cast<TremoloParser &>(*ParserList[tremolo]);
}

/** Returns reference to the output XyzParser, adds if not present.
 * \return reference to the output XyzParser
 */
XyzParser &FormatParserStorage::getXyz()
{
  if (!ParserPresent[xyz])
    addXyz();
  return dynamic_cast<XyzParser &>(*ParserList[xyz]);
}



CONSTRUCT_SINGLETON(FormatParserStorage)
