/*
 * ObserverTest.cpp
 *
 *  Created on: Jan 19, 2010
 *      Author: crueger
 */

#include "ObserverTest.hpp"

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "Patterns/Observer.hpp"
#include "Helpers/Assert.hpp"

#include <iostream>

using namespace std;

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( ObserverTest );

/******************* Test stubs ************************/

class UpdateCountObserver : public Observer {
public:
  UpdateCountObserver() :
    updates(0)
  {};
  void update(Observable *publisher){
    updates++;
  }
  void subjectKilled(Observable *publisher) {
  }
  int updates;
};

class SimpleObservable : public Observable {
public:
  void changeMethod() {
    OBSERVE;
    int i = 0;
    i++;
  }
};

class CallObservable : public Observable {
public:
  void changeMethod1() {
    OBSERVE;
    int i = 0;
    i++;
  }

  void changeMethod2() {
    OBSERVE;
    int i = 0;
    i++;
    changeMethod1();
  }
};

class BlockObservable : public Observable {
public:
  void changeMethod1(){
    OBSERVE;
    // test if we report correctly as blocked
    CPPUNIT_ASSERT(isBlocked());
  }

  void changeMethod2(){
    OBSERVE;
    internalMethod1();
    internalMethod2();
  }

  void internalMethod1(){
    // we did not block, but our caller did...
    // see if this is found
    CPPUNIT_ASSERT(isBlocked());
  }

  void internalMethod2(){
    OBSERVE;
    // Both this method and the caller do block
    // Does the reporting still work as expected?
    CPPUNIT_ASSERT(isBlocked());
  }

  void noChangeMethod(){
    // No Block introduced here
    // reported correctely?
    CPPUNIT_ASSERT(!isBlocked());
  }
};

class SuperObservable : public Observable {
public:
  SuperObservable(){
    subObservable = new SimpleObservable();
    subObservable->signOn(this);
  }
  ~SuperObservable(){
    delete subObservable;
  }
  void changeMethod() {
    OBSERVE;
    int i = 0;
    i++;
    subObservable->changeMethod();
  }
  SimpleObservable *subObservable;
};

/******************* actuall tests ***************/

void ObserverTest::setUp() {
  ASSERT_DO(Assert::Throw);
  simpleObservable1 = new SimpleObservable();
  simpleObservable2 = new SimpleObservable();
  callObservable = new CallObservable();
  superObservable = new SuperObservable();
  blockObservable = new BlockObservable();

  observer1 = new UpdateCountObserver();
  observer2 = new UpdateCountObserver();
  observer3 = new UpdateCountObserver();
  observer4 = new UpdateCountObserver();
}

void ObserverTest::tearDown() {
  delete simpleObservable1;
  delete simpleObservable2;
  delete callObservable;
  delete superObservable;

  delete observer1;
  delete observer2;
  delete observer3;
  delete observer4;
}

void ObserverTest::doesUpdateTest()
{
  simpleObservable1->signOn(observer1);
  simpleObservable1->signOn(observer2);
  simpleObservable1->signOn(observer3);

  simpleObservable2->signOn(observer2);
  simpleObservable2->signOn(observer4);

  simpleObservable1->changeMethod();
  CPPUNIT_ASSERT_EQUAL( 1, observer1->updates );
  CPPUNIT_ASSERT_EQUAL( 1, observer2->updates );
  CPPUNIT_ASSERT_EQUAL( 1, observer3->updates );
  CPPUNIT_ASSERT_EQUAL( 0, observer4->updates );

  simpleObservable1->signOff(observer3);

  simpleObservable1->changeMethod();
  CPPUNIT_ASSERT_EQUAL( 2, observer1->updates );
  CPPUNIT_ASSERT_EQUAL( 2, observer2->updates );
  CPPUNIT_ASSERT_EQUAL( 1, observer3->updates );
  CPPUNIT_ASSERT_EQUAL( 0, observer4->updates );

  simpleObservable2->changeMethod();
  CPPUNIT_ASSERT_EQUAL( 2, observer1->updates );
  CPPUNIT_ASSERT_EQUAL( 3, observer2->updates );
  CPPUNIT_ASSERT_EQUAL( 1, observer3->updates );
  CPPUNIT_ASSERT_EQUAL( 1, observer4->updates );
}


void ObserverTest::doesBlockUpdateTest() {
  callObservable->signOn(observer1);

  callObservable->changeMethod1();
  CPPUNIT_ASSERT_EQUAL( 1, observer1->updates );

  callObservable->changeMethod2();
  CPPUNIT_ASSERT_EQUAL( 2, observer1->updates );
}

void ObserverTest::doesSubObservableTest() {
  superObservable->signOn(observer1);
  superObservable->subObservable->signOn(observer2);

  superObservable->subObservable->changeMethod();
  CPPUNIT_ASSERT_EQUAL( 1, observer1->updates );
  CPPUNIT_ASSERT_EQUAL( 1, observer2->updates );

  superObservable->changeMethod();
  CPPUNIT_ASSERT_EQUAL( 2, observer1->updates );
  CPPUNIT_ASSERT_EQUAL( 2, observer2->updates );
}

void ObserverTest::doesReportTest(){
  // Actual checks are in the Stub-methods for this
  blockObservable->changeMethod1();
  blockObservable->changeMethod2();
  blockObservable->noChangeMethod();
}

void ObserverTest::CircleDetectionTest() {
  cout << endl << "Warning: the next test involved methods that can produce infinite loops." << endl;
  cout << "Errors in this methods can not be checked using the CPPUNIT_ASSERT Macros." << endl;
  cout << "Instead tests are run on these methods to see if termination is assured" << endl << endl;
  cout << "If this test does not complete in a few seconds, kill the test-suite and fix the Error in the circle detection mechanism" << endl;

  cout << endl << endl << "The following errors displayed by the observer framework can be ignored" << endl;

  // make this Observable its own subject. NEVER DO THIS IN ACTUAL CODE
  simpleObservable1->signOn(simpleObservable1);
  CPPUNIT_ASSERT_THROW(simpleObservable1->changeMethod(),Assert::AssertionFailure);

  // more complex test
  simpleObservable1->signOff(simpleObservable1);
  simpleObservable1->signOn(simpleObservable2);
  simpleObservable2->signOn(simpleObservable1);
  CPPUNIT_ASSERT_THROW(simpleObservable1->changeMethod(),Assert::AssertionFailure);
  simpleObservable1->signOff(simpleObservable2);
  simpleObservable2->signOff(simpleObservable1);
  // when we reach this line, although we broke the DAG assumption the circle check works fine
  CPPUNIT_ASSERT(true);
}
