/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2013 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * FitPartialChargesAction.cpp
 *
 *  Created on: Jul 03, 2013
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// needs to come before MemDebug due to placement new
#include <boost/archive/text_iarchive.hpp>

#include "CodePatterns/MemDebug.hpp"

#include "Atom/atom.hpp"
#include "CodePatterns/Log.hpp"
#include "Fragmentation/Exporters/ExportGraph_ToFiles.hpp"
#include "Fragmentation/Graph.hpp"
#include "World.hpp"

#include <boost/bimap.hpp>
#include <boost/bimap/multiset_of.hpp>
#include <boost/bind.hpp>
#include <boost/filesystem.hpp>
#include <boost/foreach.hpp>
#include <boost/mpl/for_each.hpp>

#include <algorithm>
#include <functional>
#include <iostream>
#include <string>

#include "Actions/PotentialAction/FitPartialChargesAction.hpp"

#include "Potentials/PartialNucleiChargeFitter.hpp"

#include "AtomIdSet.hpp"
#include "Descriptors/AtomIdDescriptor.hpp"
#include "Element/element.hpp"
#include "Element/periodentafel.hpp"
#include "Fragmentation/Homology/AtomFragmentsMap.hpp"
#include "Fragmentation/Homology/HomologyContainer.hpp"
#include "Fragmentation/Homology/HomologyGraph.hpp"
#include "Fragmentation/Summation/Containers/PartialChargesFused.hpp"
#include "Fragmentation/Summation/Containers/PartialChargesMap.hpp"
#include "Fragmentation/Summation/SetValues/SamplingGrid.hpp"
#include "Fragmentation/Summation/ZeroInstanceInitializer.hpp"
#include "FunctionApproximation/Extractors.hpp"
#include "Potentials/PartialNucleiChargeFitter.hpp"
#include "Potentials/Particles/ParticleFactory.hpp"
#include "Potentials/Particles/ParticleRegistry.hpp"
#include "Potentials/SerializablePotential.hpp"
#include "World.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "FitPartialChargesAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */

namespace detail {
  typedef std::map<KeySet, HomologyGraph> KeysetsToGraph_t;

  typedef std::map<HomologyGraph, PartialNucleiChargeFitter::charges_t> GraphFittedChargeMap_t;

  typedef std::map<atomId_t, double> fitted_charges_t;

  typedef std::map<HomologyGraph, size_t> GraphIndex_t;

  typedef std::set<size_t> AtomsGraphIndices_t;
  typedef boost::bimaps::bimap<
      boost::bimaps::multiset_of<AtomsGraphIndices_t>,
      atomId_t > GraphIndices_t;

  typedef std::map<std::set<size_t>, std::map<atomicNumber_t, std::string> > AtomParticleNames_t;

  typedef std::map<std::set<size_t>, std::string> GraphToNameMap_t;
};

static void enforceZeroTotalCharge(
    PartialNucleiChargeFitter::charges_t &_averaged_charges)
{
  double charge_sum = 0.;
  charge_sum = std::accumulate(_averaged_charges.begin(), _averaged_charges.end(), charge_sum);
  if (fabs(charge_sum) > MYEPSILON) {
    std::transform(
        _averaged_charges.begin(), _averaged_charges.end(),
        _averaged_charges.begin(),
        boost::bind(std::minus<double>(), _1, charge_sum/_averaged_charges.size()));
  }
  charge_sum = 0.;
  charge_sum = std::accumulate(_averaged_charges.begin(), _averaged_charges.end(), charge_sum);
  ASSERT( fabs(charge_sum) < MYEPSILON,
      "enforceZeroTotalCharge() - enforcing neutral net charge failed, "
      +toString(charge_sum)+" is the remaining net charge.");

  LOG(2, "DEBUG: final charges with net zero charge are " << _averaged_charges);
}

static size_t obtainAverageChargesOverFragments(
    PartialNucleiChargeFitter::charges_t &_average_charges,
    const std::pair<
      HomologyContainer::const_iterator,
      HomologyContainer::const_iterator> &_range,
    const double _radius
    )
{
  HomologyContainer::const_iterator iter = _range.first;
  if (!iter->second.containsGrids) {
    ELOG(1, "This HomologyGraph does not contain sampled grids.");
    return 0;
  }
  _average_charges.resize(iter->second.fragment.getCharges().size(), 0.);
  size_t NoFragments = 0;
  for (;
      iter != _range.second; ++iter, ++NoFragments) {
    if (!iter->second.containsGrids) {
      ELOG(2, "This HomologyGraph does not contain sampled grids,\ndid you forget to add '--store-grids 1' to AnalyseFragmentResults.");
      return 0;
    }
    const Fragment &fragment = iter->second.fragment;
  //  const double &energy = iter->second.energy;
  //  const SamplingGrid &charge = iter->second.charge_distribution;
    const SamplingGrid &potential = iter->second.potential_distribution;
    if ((potential.level == 0)
        || ((potential.begin[0] == potential.end[0])
            && (potential.begin[1] == potential.end[1])
            && (potential.begin[2] == potential.end[2]))) {
      ELOG(1, "Sampled grid contains grid made of zero points.");
      return 0;
    }

    // then we extract positions from fragment
    PartialNucleiChargeFitter::positions_t positions;
    Fragment::positions_t fragmentpositions = fragment.getPositions();
    positions.reserve(fragmentpositions.size());
    BOOST_FOREACH( Fragment::position_t pos, fragmentpositions) {
      positions.push_back( Vector(pos[0], pos[1], pos[2]) );
    }
    PartialNucleiChargeFitter fitter(potential, positions, _radius);
    fitter();
    PartialNucleiChargeFitter::charges_t return_charges = fitter.getSolutionAsCharges_t();
    LOG(2, "DEBUG: fitted charges are " << return_charges);
    std::transform(
        return_charges.begin(), return_charges.end(),
        _average_charges.begin(),
        _average_charges.begin(),
        std::plus<PartialNucleiChargeFitter::charge_t>());
  }
  if (NoFragments != 0)
    std::transform(_average_charges.begin(), _average_charges.end(),
        _average_charges.begin(),
        std::bind1st(std::multiplies<PartialNucleiChargeFitter::charge_t>(),1./(double)NoFragments)
    );
  LOG(2, "DEBUG: final averaged charges are " << _average_charges);

  return NoFragments;
}

static
std::set<KeySet> accumulateKeySetsForAtoms(
    const AtomFragmentsMap::AtomFragmentsMap_t &_atommap,
    const std::vector<const atom *> &_selected_atoms)
{
  std::set<KeySet> fragments;
  for (std::vector<const atom *>::const_iterator iter = _selected_atoms.begin();
      iter != _selected_atoms.end(); ++iter) {
    const atomId_t atomid = (*iter)->getId();
    const AtomFragmentsMap::AtomFragmentsMap_t::const_iterator atomiter =
        _atommap.find(atomid);
    if ((*iter)->getElementNo() != 1) {
      if (atomiter == _atommap.end()) {
        ELOG(2, "There are no fragments associated to " << atomid << ".");
        continue;
      }
      const AtomFragmentsMap::keysets_t &keysets = atomiter->second;
      LOG(2, "DEBUG: atom " << atomid << " has " << keysets.size() << " fragments.");
      fragments.insert( keysets.begin(), keysets.end() );
    } else {
      LOG(3, "DEBUG: Skipping atom " << atomid << " as it's hydrogen.");
    }
  }
  return fragments;
}

static
void getKeySetsToGraphMapping(
    detail::KeysetsToGraph_t &_keyset_graphs,
    detail::GraphFittedChargeMap_t &_fittedcharges_per_fragment,
    const std::set<KeySet> &_fragments,
    const AtomFragmentsMap &_atomfragments)
{
  for (std::set<KeySet>::const_iterator fragmentiter = _fragments.begin();
      fragmentiter != _fragments.end(); ++fragmentiter) {
    const KeySet &keyset = *fragmentiter;
    const AtomFragmentsMap::indices_t &forceindices = _atomfragments.getFullKeyset(keyset);
    ASSERT( !forceindices.empty(),
        "getKeySetsToGraphMapping() - force keyset to "+toString(keyset)+" is empty.");
    KeySet forcekeyset;
    forcekeyset.insert(forceindices.begin(), forceindices.end());
    forcekeyset.erase(-1);
    const HomologyGraph graph(forcekeyset);
    LOG(2, "DEBUG: Associating keyset " << forcekeyset << " with graph " << graph);
    _keyset_graphs.insert( std::make_pair(keyset, graph) );
    _fittedcharges_per_fragment.insert( std::make_pair(graph, PartialNucleiChargeFitter::charges_t()) );
  }
}

static
bool getPartialChargesForAllGraphs(
    detail::GraphFittedChargeMap_t &_fittedcharges_per_fragment,
    const HomologyContainer &_homologies,
    const double _mask_radius,
    const bool enforceZeroCharge)
{
  for (detail::GraphFittedChargeMap_t::iterator graphiter = _fittedcharges_per_fragment.begin();
      graphiter != _fittedcharges_per_fragment.end(); ++graphiter) {
    const HomologyGraph &graph = graphiter->first;
    LOG(2, "DEBUG: Now fitting charges to graph " << graph);
    const HomologyContainer::range_t range = _homologies.getHomologousGraphs(graph);
    if (range.first == range.second) {
      ELOG(0, "HomologyContainer does not contain specified fragment.");
      return false;
    }

    // fit and average partial charges over all homologous fragments
    PartialNucleiChargeFitter::charges_t &averaged_charges = graphiter->second;
    const size_t NoFragments =
        obtainAverageChargesOverFragments(averaged_charges, range, _mask_radius);
    if ((NoFragments == 0) && (range.first != range.second)) {
      ELOG(0, "Fitting for fragment "+toString(*graphiter)+" failed.");
      return false;
    }

    // make sum of charges zero if desired
    if (enforceZeroCharge)
      enforceZeroTotalCharge(averaged_charges);

    // output status info fitted charges
    LOG(2, "DEBUG: For fragment " << *graphiter << " we have fitted the following charges "
        << averaged_charges << ", averaged over " << NoFragments << " fragments.");
  }
  return true;
}

static const atom * getNonHydrogenSurrogate(const atom * const _walker)
{
  const atom * surrogate = _walker;
  if (surrogate->getElementNo() == 1) {
    // it's hydrogen, check its bonding and use its bond partner instead to request
    // keysets
    const BondList &ListOfBonds = surrogate->getListOfBonds();
    if ( ListOfBonds.size() != 1) {
      ELOG(1, "Solitary hydrogen in atom " << surrogate->getId() << " detected.");
      return _walker;
    }
    surrogate = (*ListOfBonds.begin())->GetOtherAtom(surrogate);
  }
  return surrogate;
}

static double fitAverageChargeToAtom(
    const atom * const _walker,
    const AtomFragmentsMap &_atomfragments,
    const detail::KeysetsToGraph_t &_keyset_graphs,
    const detail::GraphFittedChargeMap_t &_fittedcharges_per_fragment)
{
  const atom * const surrogate = getNonHydrogenSurrogate(_walker);
  const atomId_t walkerid = surrogate->getId();
  const AtomFragmentsMap::AtomFragmentsMap_t &atommap = _atomfragments.getMap();
  const AtomFragmentsMap::AtomFragmentsMap_t::const_iterator keysetsiter =
      atommap.find(walkerid);
  ASSERT(keysetsiter != atommap.end(),
      "fitAverageChargeToAtom() - we checked already that "+toString(walkerid)
      +" should be present!");
  const AtomFragmentsMap::keysets_t & keysets = keysetsiter->second;

  double average_charge = 0.;
  size_t NoFragments = 0;
  // go over all fragments associated to this atom
  for (AtomFragmentsMap::keysets_t::const_iterator keysetsiter = keysets.begin();
      keysetsiter != keysets.end(); ++keysetsiter) {
    const KeySet &keyset = *keysetsiter;

    const AtomFragmentsMap::indices_t &forcekeyset = _atomfragments.getFullKeyset(keyset);
    ASSERT( !forcekeyset.empty(),
        "fitAverageChargeToAtom() - force keyset to "+toString(keyset)+" is empty.");

    // find the associated charge in the charge vector
    const std::map<KeySet, HomologyGraph>::const_iterator keysetgraphiter =
        _keyset_graphs.find(keyset);
    ASSERT( keysetgraphiter != _keyset_graphs.end(),
        "fitAverageChargeToAtom() - keyset "+toString(keyset)
        +" not contained in keyset_graphs.");
    const HomologyGraph &graph = keysetgraphiter->second;
    const detail::GraphFittedChargeMap_t::const_iterator chargesiter =
        _fittedcharges_per_fragment.find(graph);
    ASSERT(chargesiter != _fittedcharges_per_fragment.end(),
        "fitAverageChargeToAtom() - no charge to "+toString(keyset)
        +" any longer present in fittedcharges_per_fragment?");
    const PartialNucleiChargeFitter::charges_t &charges = chargesiter->second;
    ASSERT( charges.size() == forcekeyset.size(),
        "fitAverageChargeToAtom() - charges "+toString(charges.size())+" and keyset "
        +toString(forcekeyset.size())+" do not have the same length?");
    PartialNucleiChargeFitter::charges_t::const_iterator chargeiter =
        charges.begin();
    const AtomFragmentsMap::indices_t::const_iterator forcekeysetiter =
        std::find(forcekeyset.begin(), forcekeyset.end(), _walker->getId());
    ASSERT( forcekeysetiter != forcekeyset.end(),
        "fitAverageChargeToAtom() - atom "+toString(_walker->getId())
        +" not contained in force keyset "+toString(forcekeyset));
    std::advance(chargeiter, std::distance(forcekeyset.begin(), forcekeysetiter));

    // and add onto charge sum
    const double & charge_in_fragment = *chargeiter;
    average_charge += charge_in_fragment;
    ++NoFragments;
  }
  // average to obtain final partial charge for this atom
  average_charge *= 1./(double)NoFragments;

  return average_charge;
}

static void addToParticleRegistry(
    const ParticleFactory &factory,
    const periodentafel &periode,
    const detail::fitted_charges_t &_fitted_charges,
    const detail::GraphIndices_t &_GraphIndices,
    detail::AtomParticleNames_t &_atom_particlenames)
{
  for (detail::fitted_charges_t::const_iterator chargeiter = _fitted_charges.begin();
      chargeiter != _fitted_charges.end(); ++chargeiter) {
    const atomId_t &atomid = chargeiter->first;
    const atom * const walker = World::getInstance().getAtom(AtomById(atomid));
    ASSERT( walker != NULL,
        "addToParticleRegistry() - atom "+toString(atomid)
        +" not present in the World?");
    const detail::GraphIndices_t::right_const_iterator graphiter =
        _GraphIndices.right.find(atomid);
    ASSERT(graphiter != _GraphIndices.right.end(),
        "addToParticleRegistry() - atom #"+toString(atomid)
        +" not contained in GraphIndices.");
    const detail::AtomParticleNames_t::iterator nameiter =
        _atom_particlenames.find(graphiter->second);
    const atomicNumber_t elementno = walker->getElementNo();
    std::string name;
    if ((nameiter != _atom_particlenames.end()) && (nameiter->second.count(elementno))) {
        name = (nameiter->second)[elementno];
    } else {
      if (nameiter == _atom_particlenames.end())
        _atom_particlenames.insert(
            std::make_pair(graphiter->second, std::map<atomicNumber_t, std::string>()) );
      const double &charge = chargeiter->second;
      name = Particle::findFreeName(periode, elementno);
      _atom_particlenames[graphiter->second][elementno] = name;
      LOG(1, "INFO: Adding particle " << name << " for atom "
          << *walker << " with element " << elementno << ", charge " << charge);
      factory.createInstance(name, elementno, charge);
    }
  }
}

static bool isNotHydrogen(const atom * const _atom)
{
  return (_atom->getElementNo() != (atomicNumber_t) 1);
}

static struct KeySetSizeComp {
  bool operator() (const KeySet &a, const KeySet &b) { return a.size()<b.size(); }
} keyset_comparator;

#include <boost/fusion/sequence.hpp>
#include <boost/mpl/for_each.hpp>

#include "Fragmentation/Summation/AllLevelOrthogonalSummator.hpp"
#include "Fragmentation/Summation/IndexSet.hpp"
#include "Fragmentation/Summation/IndexSetContainer.hpp"
#include "Fragmentation/Summation/SubsetMap.hpp"
#include "Fragmentation/Summation/Containers/PartialChargesFused.hpp"
#include "Fragmentation/Summation/Containers/PartialChargesMap.hpp"
#include "Fragmentation/Summation/SetValues/IndexedPartialCharges.hpp"

ActionState::ptr PotentialFitPartialChargesAction::performCall()
{
  // check for selected atoms
  const World &world = World::getConstInstance();
  const std::vector<const atom *> selected_atoms = world.getSelectedAtoms();
  if (selected_atoms.empty()) {
    STATUS("There are no atoms selected for fitting partial charges to.");
    return Action::failure;
  }
  LOG(3, "There are  " << selected_atoms.size() << " selected atoms.");

  /// obtain possible fragments to each selected atom
  const AtomFragmentsMap &atomfragments = AtomFragmentsMap::getConstInstance();
  if (!atomfragments.checkCompleteness()) {
    ELOG(0, "AtomFragmentsMap failed internal consistency check, missing forcekeysets?");
    return Action::failure;
  }
  const std::set<KeySet> fragments =
      accumulateKeySetsForAtoms( atomfragments.getMap(), selected_atoms);
  const size_t NoNonHydrogens =
      std::count_if(selected_atoms.begin(), selected_atoms.end(), isNotHydrogen);
  if (fragments.size() < NoNonHydrogens) {
    ELOG(0, "Obtained fewer fragments than there are atoms, has AtomFragments been loaded?");
    return Action::failure;
  }

  // reduce given fragments to homologous graphs to avoid multiple fittings
  detail::KeysetsToGraph_t keyset_graphs;
  detail::GraphFittedChargeMap_t fittedcharges_per_fragment;
  getKeySetsToGraphMapping(keyset_graphs, fittedcharges_per_fragment, fragments, atomfragments);

  /// then go through all fragments and get partial charges for each
  const HomologyContainer &homologies = World::getInstance().getHomologies();
  const bool status = getPartialChargesForAllGraphs(
      fittedcharges_per_fragment,
      homologies,
      params.radius.get(),
      params.enforceZeroCharge.get());
  if (!status)
    return Action::failure;

  /// obtain average charge for each atom the fitted charges over all its fragments
  detail::fitted_charges_t fitted_charges;
//  for (std::vector<const atom *>::const_iterator atomiter = selected_atoms.begin();
//      atomiter != selected_atoms.end(); ++atomiter) {
//    const atomId_t walkerid = (*atomiter)->getId();
//    const double average_charge = fitAverageChargeToAtom(
//        *atomiter, atomfragments, keyset_graphs, fittedcharges_per_fragment);
//
//    if (average_charge != 0.) {
//      LOG(2, "DEBUG: For atom " << **atomiter << " we have an average charge of "
//          << average_charge);
//
//      fitted_charges.insert( std::make_pair(walkerid, average_charge) );
//    }
//  }
  {
    // TODO:: convert the data into boost::fusion::map format
    std::map<JobId_t, PartialChargesMap_t> PartialCharges_fused;
    std::vector<IndexSet> indexsets;
    {
      JobId_t counter = 0;
      for (std::set<KeySet>::const_iterator iter = fragments.begin();
          iter != fragments.end(); ++iter) {
        const KeySet &currentset = *iter;
        // place as IndexSet into container
        {
          IndexSet tempset;
          tempset.insert(currentset.begin(), currentset.end());
          indexsets.push_back(tempset);
        }
        const detail::KeysetsToGraph_t::const_iterator graphiter = keyset_graphs.find(currentset);
        if (graphiter == keyset_graphs.end()) {
          ELOG(1, "Could not find graph to keyset " << currentset);
          return Action::failure;
        }
        const HomologyGraph &currentgraph = graphiter->second;
        const detail::GraphFittedChargeMap_t::const_iterator fragmentiter =
            fittedcharges_per_fragment.find(currentgraph);
        if (fragmentiter == fittedcharges_per_fragment.end()) {
          ELOG(1, "Could not find fragment to graph " << currentgraph);
          return Action::failure;
        }
        const PartialNucleiChargeFitter::charges_t &charges = fragmentiter->second;
        PartialChargesMap_t chargemap;
        const AtomFragmentsMap::indices_t &full_currentset = atomfragments.getFullKeyset(currentset);
        IndexedPartialCharges::indices_t indices(full_currentset.begin(), full_currentset.end());
        IndexedPartialCharges::values_t values(charges.begin(), charges.end());
        LOG(3, "Inserting " << indices << " with charges " << charges << " into chargemap instance.");
        boost::fusion::at_key<PartialChargesFused::partial_charges_t>(chargemap) =
            IndexedPartialCharges(indices, values);
        PartialCharges_fused.insert( std::make_pair(counter++, chargemap) );
      }
    }
    ASSERT( indexsets.size() == fragments.size(),
        "PotentialFitPartialChargesAction::performCall() - not all fragments' keysets were created?");

    // prepare index set hierarchy
    std::map< JobId_t, size_t > IdentityLookup;
    size_t MaxLevel = std::max_element(fragments.begin(), fragments.end(), keyset_comparator)->size();
    LOG(3, "The maximum level is " << MaxLevel);
    size_t max_indices = fragments.size();
    LOG(3, "There are  " << max_indices << " for the selected atoms.");
    {
      for (size_t index = 0; index < max_indices; ++index)
        IdentityLookup.insert( std::make_pair( (JobId_t)index, index ) );
    }
    IndexSetContainer::ptr container(new IndexSetContainer(indexsets));
    SubsetMap::ptr subsetmap(new SubsetMap(*container));

    // and sum up
    PartialChargesMap_t ZeroInstances;
    ZeroInstanceInitializer<PartialChargesMap_t> initZeroInstance(ZeroInstances);
    boost::mpl::for_each<PartialChargesVector_t>(boost::ref(initZeroInstance));
    //!> results per level of summed up partial charges
    std::vector<PartialChargesMap_t> Result_PartialCharges_fused(MaxLevel);
    //!> results per index set in terms of value and contribution
    std::map<
        IndexSet::ptr,
        std::pair<PartialChargesMap_t, PartialChargesMap_t> > Result_perIndexSet_PartialCharges;
    AllLevelOrthogonalSummator<PartialChargesMap_t> partialchargeSummer(
                subsetmap,
                PartialCharges_fused,
                container->getContainer(),
                IdentityLookup,
                Result_PartialCharges_fused,
                Result_perIndexSet_PartialCharges,
                ZeroInstances);
    boost::mpl::for_each<PartialChargesVector_t>(boost::ref(partialchargeSummer));

    // TODO: place results into fitted_charges
    const IndexedPartialCharges::indexedvalues_t indexed_partial_charges =
        boost::fusion::at_key<PartialChargesFused::partial_charges_t>(
            Result_PartialCharges_fused.back()
            ).getValues();
    for (IndexedPartialCharges::indexedvalues_t::const_iterator iter = indexed_partial_charges.begin();
        iter != indexed_partial_charges.end(); ++iter)
    fitted_charges.insert( std::make_pair( iter->first, iter->second.charge));
    LOG(3, "Summation has brought forth the following charges per atom index: " << fitted_charges);
  }

  /// make Particles be used for every atom that was fitted on the same number of graphs
  detail::GraphIndex_t GraphIndex;
  size_t index = 0;
  for (HomologyContainer::const_key_iterator iter = homologies.key_begin();
      iter != homologies.key_end(); iter = homologies.getNextKey(iter)) {
    GraphIndex.insert( std::make_pair( *iter, index++));
  }
  LOG(2, "DEBUG: There are " << index << " unique graphs in the homology container.");

  // go through every non-hydrogen atom, get all graphs, convert to GraphIndex and store
  detail::GraphIndices_t GraphIndices;
  const AtomFragmentsMap::AtomFragmentsMap_t &atommap = atomfragments.getMap();
  for (std::vector<const atom *>::const_iterator atomiter = selected_atoms.begin();
      atomiter != selected_atoms.end(); ++atomiter) {
    // use the non-hydrogen here
    const atomId_t walkerid = (*atomiter)->getId();
    const atomId_t surrogateid = getNonHydrogenSurrogate(*atomiter)->getId();
    if (surrogateid != walkerid)
      continue;
    const AtomFragmentsMap::AtomFragmentsMap_t::const_iterator keysetsiter =
        atommap.find(walkerid);
    ASSERT(keysetsiter != atommap.end(),
        "PotentialFitPartialChargesAction::performCall() - we checked already that "
        +toString(surrogateid)+" should be present!");
    const AtomFragmentsMap::keysets_t & keysets = keysetsiter->second;

    // go over all fragments associated to this atom
    detail::AtomsGraphIndices_t AtomsGraphIndices;
    for (AtomFragmentsMap::keysets_t::const_iterator keysetsiter = keysets.begin();
        keysetsiter != keysets.end(); ++keysetsiter) {
      const KeySet &keyset = *keysetsiter;
      const std::map<KeySet, HomologyGraph>::const_iterator keysetgraphiter =
          keyset_graphs.find(keyset);
      ASSERT( keysetgraphiter != keyset_graphs.end(),
          "PotentialFitPartialChargesAction::performCall() - keyset "+toString(keyset)
          +" not contained in keyset_graphs.");
      const HomologyGraph &graph = keysetgraphiter->second;
      const detail::GraphIndex_t::const_iterator indexiter = GraphIndex.find(graph);
      ASSERT( indexiter != GraphIndex.end(),
          "PotentialFitPartialChargesAction::performCall() - graph "+toString(graph)
          +" not contained in GraphIndex.");
      AtomsGraphIndices.insert( indexiter->second );
    }

    GraphIndices.insert( detail::GraphIndices_t::value_type(AtomsGraphIndices, walkerid) );

    LOG(2, "DEBUG: Atom #" << walkerid << "," << *atomiter << ". has graph indices "
        << AtomsGraphIndices);
  }
  // then graphs from non-hydrogen bond partner for all hydrogens
  for (std::vector<const atom *>::const_iterator atomiter = selected_atoms.begin();
      atomiter != selected_atoms.end(); ++atomiter) {
    // use the non-hydrogen here
    const atomId_t walkerid = (*atomiter)->getId();
    const atomId_t surrogateid = getNonHydrogenSurrogate((*atomiter))->getId();
    if (surrogateid == walkerid)
      continue;
    detail::GraphIndices_t::right_const_iterator graphiter = GraphIndices.right.find(surrogateid);
    ASSERT( graphiter != GraphIndices.right.end(),
        "PotentialFitPartialChargesAction::performCall() - atom #"+toString(surrogateid)
        +" not contained in GraphIndices.");
    const detail::AtomsGraphIndices_t &AtomsGraphIndices = graphiter->second;
    GraphIndices.insert( detail::GraphIndices_t::value_type(AtomsGraphIndices, walkerid) );
    LOG(2, "DEBUG: Hydrogen #" << walkerid << ", " << *atomiter
        << ", has graph indices " << AtomsGraphIndices);
  }

  /// place all fitted charges into ParticleRegistry
  detail::AtomParticleNames_t atom_particlenames;
  addToParticleRegistry(
      ParticleFactory::getConstInstance(),
      *World::getInstance().getPeriode(),
      fitted_charges,
      GraphIndices,
      atom_particlenames);
  for (World::AtomSelectionIterator atomiter = World::getInstance().beginAtomSelection();
      atomiter != World::getInstance().endAtomSelection(); ++atomiter) {
    atom * const walker = atomiter->second;
    const atomId_t walkerid = atomiter->first;
    const detail::GraphIndices_t::right_const_iterator graphiter =
        GraphIndices.right.find(walkerid);
    ASSERT( graphiter != GraphIndices.right.end(),
        "PotentialFitPartialChargesAction::performCall() - cannot find "
        +toString(walkerid)+" in GraphIndices.");
    const detail::AtomsGraphIndices_t &graphindex = graphiter->second;
    const detail::AtomParticleNames_t::const_iterator particlesetiter =
        atom_particlenames.find(graphindex);
    ASSERT( particlesetiter != atom_particlenames.end(),
        "PotentialFitPartialChargesAction::performCall() - cannot find "
        +toString(graphindex)+" in atom_particlenames.");
    const std::map<atomicNumber_t, std::string>::const_iterator nameiter =
        particlesetiter->second.find(walker->getElementNo());
    ASSERT( nameiter != particlesetiter->second.end(),
        "PotentialFitPartialChargesAction::performCall() - ");
    walker->setParticleName(nameiter->second);
    LOG(1, "INFO: atom " << *walker << " received the following particle "
        << walker->getParticleName());
  }

  return Action::success;
}

ActionState::ptr PotentialFitPartialChargesAction::performUndo(ActionState::ptr _state) {
  return Action::success;
}

ActionState::ptr PotentialFitPartialChargesAction::performRedo(ActionState::ptr _state){
  return Action::success;
}

bool PotentialFitPartialChargesAction::canUndo() {
  return false;
}

bool PotentialFitPartialChargesAction::shouldUndo() {
  return false;
}
/** =========== end of function ====================== */
