/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2013 University of Bonn. All rights reserved.
 * Copyright (C)  2013 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * FitPotentialAction.cpp
 *
 *  Created on: Apr 09, 2013
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// needs to come before MemDebug due to placement new
#include <boost/archive/text_iarchive.hpp>

#include "CodePatterns/MemDebug.hpp"

#include <algorithm>
#include <boost/bind.hpp>
#include <boost/filesystem.hpp>
#include <boost/foreach.hpp>
#include <map>
#include <string>
#include <sstream>

#include "Actions/PotentialAction/FitPotentialAction.hpp"

#include "CodePatterns/Log.hpp"

#include "Element/element.hpp"
#include "Fragmentation/Homology/HomologyContainer.hpp"
#include "Fragmentation/Homology/HomologyGraph.hpp"
#include "Fragmentation/Summation/SetValues/Fragment.hpp"
#include "Potentials/EmpiricalPotential.hpp"
#include "Potentials/PotentialFactory.hpp"
#include "Potentials/PotentialRegistry.hpp"
#include "Potentials/PotentialTrainer.hpp"
#include "Potentials/SerializablePotential.hpp"
#include "World.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "FitPotentialAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */

ActionState::ptr PotentialFitPotentialAction::performCall() {
  // fragment specifies the homology fragment to use
  SerializablePotential::ParticleTypes_t fragmentnumbers =
      PotentialTrainer::getNumbersFromElements(params.fragment.get());

  // either charges and a potential is specified or a file
  if (params.charges.get().empty()) {
    STATUS("No charges given!");
    return Action::failure;
  } else {
    // charges specify the potential type
    SerializablePotential::ParticleTypes_t chargenumbers =
        PotentialTrainer::getNumbersFromElements(params.charges.get());

    LOG(0, "STATUS: I'm training now a " << params.potentialtype.get()
        << " potential on charges " << chargenumbers << " on data from World's homologies.");

    // register desired potential and an additional constant one
    {
      EmpiricalPotential *potential =
          PotentialFactory::getInstance().createInstance(
              params.potentialtype.get(),
              chargenumbers);
      // check whether such a potential already exists
      const std::string potential_name = potential->getName();
      if (PotentialRegistry::getInstance().isPresentByName(potential_name)) {
        delete potential;
        potential = PotentialRegistry::getInstance().getByName(potential_name);
      } else
        PotentialRegistry::getInstance().registerInstance(potential);
    }
    {
      EmpiricalPotential *constant =
          PotentialFactory::getInstance().createInstance(
              std::string("constant"),
              SerializablePotential::ParticleTypes_t());
      // check whether such a potential already exists
      const std::string constant_name = constant->getName();
      if (PotentialRegistry::getInstance().isPresentByName(constant_name)) {
        delete constant;
        constant = PotentialRegistry::getInstance().getByName(constant_name);
      } else
        PotentialRegistry::getInstance().registerInstance(constant);
    }
  }

  // parse homologies into container
  const HomologyContainer &homologies = World::getInstance().getHomologies();

  // then we ought to pick the right HomologyGraph ...
  const HomologyGraph graph =
      PotentialTrainer::getFirstGraphwithSpecifiedElements(homologies,fragmentnumbers);
  if (graph != HomologyGraph()) {
    LOG(1, "First representative graph containing fragment "
        << fragmentnumbers << " is " << graph << ".");
  } else {
    STATUS("Specific fragment "+toString(fragmentnumbers)+" not found in homologies!");
    return Action::failure;
  }

  // for debugging we list all matching fragments
  HomologyContainer::range_t graphrange = homologies.getHomologousGraphs(graph);
  LOG(1, "INFO: Listing all matching homologous graphs ...");
  for (HomologyContainer::container_t::const_iterator iter =
      graphrange.first; iter != graphrange.second; ++iter) {
    LOG(1, "INFO: graph " << iter->first
        << " has Fragment " << iter->second.fragment
        << ", associated energy " << iter->second.energy
        << ", and sampled grid integral " << iter->second.charge_distribution.integral()
        << ".");
  }

  // training
  PotentialTrainer trainer;
  const bool status = trainer(
      homologies,
      graph,
      params.training_file.get(),
      params.max_iterations.get(),
      params.threshold.get(),
      params.best_of_howmany.get());
  if (!status) {
    STATUS("No required parameter derivatives for a box constraint minimization known.");
    return Action::failure;
  }

  return Action::success;
}

ActionState::ptr PotentialFitPotentialAction::performUndo(ActionState::ptr _state) {
  return Action::success;
}

ActionState::ptr PotentialFitPotentialAction::performRedo(ActionState::ptr _state){
  return Action::success;
}

bool PotentialFitPotentialAction::canUndo() {
  return false;
}

bool PotentialFitPotentialAction::shouldUndo() {
  return false;
}
/** =========== end of function ====================== */
