/*
 * TextWindow.cpp
 *
 *  Created on: Jan 7, 2010
 *      Author: crueger
 */

#include "Helpers/MemDebug.hpp"

#include <boost/bind.hpp>

#include "Menu/Menu.hpp"
#include "Menu/TextMenu.hpp"
#include "Menu/ActionMenuItem.hpp"
#include "Menu/SeperatorItem.hpp"
#include "Menu/DisplayMenuItem.hpp"
#include "Menu/SubMenuItem.hpp"
#include "TextUI/TextStatusIndicator.hpp"
#include "TextUI/TextWindow.hpp"
#include "Actions/MapOfActions.hpp"
#include "Actions/MethodAction.hpp"
#include "Actions/ErrorAction.hpp"
#include "Actions/ActionRegistry.hpp"
#include "Parser/ChangeTracker.hpp"
#include "Views/StreamStringView.hpp"
#include "Views/MethodStringView.hpp"
#include "Helpers/MemDebug.hpp"

#include "defs.hpp"
#include "log.hpp"
#include "verbose.hpp"

// all needed due to config::SaveAll()
#include "config.hpp"
#include "periodentafel.hpp"

// config::SaveAll() and enumerate()
#include "molecule.hpp"

#include <iostream>
#include <map>

// TODO: see what code can be moved to a base class for Graphic and Text Windows
TextWindow::TextWindow()
{
  map <std::string, TextMenu *> NametoTextMenuMap;

  // populate all actions
  MapOfActions::getInstance().populateActions();

  // build the main menu
  main_menu = new TextMenu(Log() << Verbose(0), "Main Menu");

  moleculeView = new StreamStringView(boost::bind(&MoleculeListClass::Enumerate,World::getInstance().getMolecules(),_1));
  new DisplayMenuItem(main_menu,moleculeView,"Molecule List");

  new SeperatorItem(main_menu);

  Action* undoAction = ActionRegistry::getInstance().getActionByName("Undo");
  new ActionMenuItem('u',"Undo last operation",main_menu,undoAction);

  Action* redoAction = ActionRegistry::getInstance().getActionByName("Redo");
  new ActionMenuItem('r',"Redo last operation",main_menu,redoAction);

  new SeperatorItem(main_menu);

  Action *setMoleculeAction = new MethodAction("setMoleculeAction",boost::bind(&MoleculeListClass::flipChosen,World::getInstance().getMolecules()));
  new ActionMenuItem('a',"set molecule (in)active",main_menu,setMoleculeAction);

  TextMenu *Menu = NULL;
  std::set <char> ShortcutList;
  for(map<std::string, pair<std::string,std::string> >::iterator iter = MapOfActions::getInstance().MenuDescription.begin(); iter != MapOfActions::getInstance().MenuDescription.end(); ++iter) {
    Menu = new TextMenu(Log() << Verbose(0), iter->second.second);
    NametoTextMenuMap.insert( pair <std::string, TextMenu *> (iter->first, Menu) );
    new SubMenuItem(getSuitableShortForm(ShortcutList,iter->first),iter->second.first.c_str(),main_menu,Menu);
  }

  new SeperatorItem(main_menu);

  Action *saveConfigAction = ActionRegistry::getInstance().getActionByName("output");
  new ActionMenuItem('s',"save current setup to config files",main_menu,saveConfigAction);

  quitAction = new MethodAction("quitAction",boost::bind(&TextMenu::doQuit,main_menu),false);
  new ActionMenuItem('q',"quit",main_menu,quitAction);

  // go through all menus and create them
  for (map <std::string, TextMenu *>::iterator MenuRunner = NametoTextMenuMap.begin(); MenuRunner != NametoTextMenuMap.end(); ++MenuRunner) {
    cout << "Creating Menu " << MenuRunner->first << "." << endl;
    populateMenu(MenuRunner->second, MenuRunner->first);
  }

  // Add status indicators etc...

  statusIndicator = new TextStatusIndicator();
}

TextWindow::~TextWindow()
{
  delete quitAction;
  delete moleculeView;
  delete statusIndicator;
  delete main_menu;
}

void TextWindow::display() {
  main_menu->display();
}

char TextWindow::getSuitableShortForm(std::set <char> &ShortcutList, const std::string name) const
{
  for (std::string::const_iterator CharRunner = name.begin(); CharRunner != name.end(); ++CharRunner) {
    if (ShortcutList.find(*CharRunner) == ShortcutList.end())
      return *CharRunner;
  }
  DoeLog(1) && (eLog() << Verbose(1) << "Could not find a suitable shortform for TextWindow::getSuitableShortForm()." << endl);
  return ((char)(ShortcutList.size() % 10) + '0');
}

void TextWindow::populateMenu(TextMenu* Menu, const  std::string &MenuName)
{
  Action *ActionItem = NULL;
  set <char> ShortcutList;
  // through all actions for this menu
  pair < multimap <std::string, std::string>::iterator, multimap <std::string, std::string>::iterator > MenuActions = MapOfActions::getInstance().MenuContainsActionMap.equal_range(MenuName);
  for (multimap <std::string, std::string>::const_iterator MenuRunner = MenuActions.first; MenuRunner != MenuActions.second; ++MenuRunner) {
    cout << " Adding " << MenuRunner->second << " to submenu " << MenuName << endl;
    ActionItem = ActionRegistry::getInstance().getActionByName(MenuRunner->second);
    new ActionMenuItem(getSuitableShortForm(ShortcutList, MenuRunner->second),MapOfActions::getInstance().getDescription(MenuRunner->second).c_str(),Menu,ActionItem);
  }
  // finally add default quit item
  Action *returnFromAction = new TextMenu::LeaveAction(Menu);
  MenuItem *returnFromItem = new ActionMenuItem('q',"return to Main menu",Menu,returnFromAction);
  Menu->addDefault(returnFromItem);
}
