/*
 * Extractors.hpp
 *
 *  Created on: 15.10.2012
 *      Author: heber
 */

#ifndef TRAININGDATA_EXTRACTORS_HPP_
#define TRAININGDATA_EXTRACTORS_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Fragmentation/SetValues/Fragment.hpp"
#include "FunctionApproximation/FunctionModel.hpp"

class Fragment;

/** Namespace containing all simple extractor functions.
 *
 */
namespace Extractors {
  typedef Fragment::charges_t::const_iterator chargeiter_t;
  typedef std::vector<chargeiter_t> chargeiters_t;

  typedef size_t count_t;
  typedef Fragment::charge_t element_t;
  typedef std::map< element_t, count_t> elementcounts_t;
  typedef std::map< element_t, chargeiters_t > elementtargets_t;

  /** Namespace for some internal helper functions.
   *
   */
  namespace _detail {
    /** Gather all positions from the same aligned vector of charges.
     *
     * Basically, we filter the positions indicated by the targets but
     * from a different vector that has the same layout.
     *
     * \param positions positions to search
     * \param charges charges to match with \targets
     * \param targets iterators on charges
     * \return filtered positions
     */
    Fragment::positions_t gatherPositionsFromCharges(
        const Fragment::positions_t &positions,
        const Fragment::charges_t &charges,
        const chargeiters_t targets
        );

    /** Counts all same elements in the vector and places into map of elements.
     *
     * \param elements vector of elements
     * \return count of same element in vector
     */
    elementcounts_t getElementCounts(
        const Fragment::charges_t elements
        );

    /** Gather iterators to the elements related to the desired elementcounts.
     *
     * \param charges charges wherein to search for the elements
     * \param elementcounts number of desired hits per element
     * \return iterators equal to the initial vector of elements
     */
    elementtargets_t convertElementcountsToTargets(
        const Fragment::charges_t &charges,
        const elementcounts_t &elementcounts
        );

    /** Convert the alignment back to as it was in the original vector.
     *
     * We lost the information by storing it in a map. Hence, we need this
     * final step.
     *
     * \param elementtargets targets as they are in the map \a elementcounts
     * \param elements the original order of the elements
     * \param elementcounts the count per element for debugging checks
     * \return vector of targets in the order as they are in \a element
     */
    chargeiters_t realignElementtargets(
        const elementtargets_t &elementtargets,
        const Fragment::charges_t elements,
        const elementcounts_t &elementcounts
        );
  }

  /** Gather all distances from a given set of positions.
   *
   * \param positions vector of positions
   * \param globalid index to associated in argument_t with
   * \return vector of argument_ , each with a distance
   */
  FunctionModel::arguments_t
  gatherAllDistanceArguments(
      const Fragment::positions_t &positions,
      const size_t globalid);

  /** Gather all distances from a given set of positions.
   *
   *  Here, we only return one of the two equal distances.
   *
   * \param positions vector of positions
   * \param globalid index to associated in argument_t with
   * \return vector of argument_ , each with a distance
   */
  FunctionModel::arguments_t
  gatherAllSymmetricDistanceArguments(
      const Fragment::positions_t &positions,
      const size_t globalid);

  /** Simple extractor of all unique pair distances of a given \a fragment.
   *
   * \param fragment fragment with all nuclei positions
   * \param index index refers to the index within the global set of configurations
   * \return vector of of argument_t containing all found distances
   */
  inline FunctionModel::arguments_t gatherAllDistances(
      const Fragment& fragment,
      const size_t index
      ) {
    // get distance out of Fragment
    return gatherAllDistanceArguments(fragment.getPositions(), index);
  }

  /** Gather all positions associated to the matching \a elements.
   *
   * \param fragment fragment with all nuclei positions
   * \param elements tuple of desired elements
   * \return vector of positions_t containing
   */
  Fragment::positions_t gatherDistanceOfTuples(
      const Fragment& fragment,
      const Fragment::charges_t elements
      );

  /** Reorder arguments by increasing distance.
   *
   * \param args arguments to reorder
   * \return reordered args
   */
  FunctionModel::arguments_t reorderArgumentsByIncreasingDistance(
      const FunctionModel::arguments_t &args
      );

}; /* namespace Extractors */


#endif /* TRAININGDATA_EXTRACTORS_HPP_ */
