/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 *
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * ShapeFactory.cpp
 *
 *  Created on: Sep 5, 2012
 *      Author: ankele
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "ShapeFactory.hpp"

#include "BaseShapes.hpp"
#include "ShapeOps.hpp"

#include "CodePatterns/Singleton_impl.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "LinearAlgebra/RealSpaceMatrix.hpp"

const char *ShapeFactory::shapeNames[ShapeFactory::SHAPETYPES_MAX] = {"nowhere", "everywhere", "sphere", "cube", "cylinder"};

ShapeFactory::ShapeFactory()
{
  // Create map (type -> name).
  for (int i=0; i<SHAPETYPES_MAX; i++)
    shapeNameMap.insert(std::pair<ShapeType, std::string>((ShapeType)i, shapeNames[i]));

  // Create inverse map.
  for (int i=0; i<SHAPETYPES_MAX; i++)
    nameShapeMap.insert(std::pair<std::string, ShapeType>(shapeNames[i], (ShapeType)i));

  // Init state.
  type = SPHERE;
  translation.Zero();
  stretch.Zero();
  for (int i=0; i<NDIM; i++)
    angle[i] = 0;
}

ShapeFactory::ShapeType ShapeFactory::getShapeByName(const std::string& name)
{
  NameShapeMap::iterator iter = nameShapeMap.find(name);
  ASSERT(iter != nameShapeMap.end(),
      "ShapeFactory::getShapeByName() - unknown name: "+name+".");
  return iter->second;
}

std::string ShapeFactory::getShapeName(ShapeType type)
{
  ShapeNameMap::iterator iter = shapeNameMap.find(type);
  ASSERT(iter != shapeNameMap.end(),
      "ShapeFactory::getShapeName() - unknown type: "+toString(type)+".");
  return iter->second;
}

ShapeFactory::~ShapeFactory()
{
}


void ShapeFactory::setStretch(const Vector &stretch)
{
  ASSERT(!stretch.IsZero(),
      "ShapeFactory::setStretch() - stretch is zero.");
  this->stretch = stretch;
}

Shape ShapeFactory::produce() const
{
  // Create the basic shape.
  Shape s = Nowhere();
  if (type == NOWHERE){
    s = Nowhere();
  }else if (type == EVERYWHERE){
    s = Everywhere();
  }else if (type == CUBE){
    s = Cuboid();
  }else if (type == SPHERE){
    s = Sphere();
  }else if (type == CYLINDER){
    s = Cylinder();
  }else{
    ASSERT(false,
        "ShapeFactory::produce - unhandled shape type: "+toString(type)+".");
  }

  // Transform (if necessary).
  if (stretch != Vector(1., 1., 1.))
    s = ::stretch(s, stretch);
  for (int i=0; i<NDIM; i++)
    if (angle[i] != 0){
      RealSpaceMatrix rotation;
      rotation.setRotation(angle);
      s = transform(s, rotation);
      break;
    }
  if (!translation.IsZero())
    s = translate(s, translation);

  return s;
}

CONSTRUCT_SINGLETON(ShapeFactory)
