/*
 * Psi3Parser_Parameters.hpp
 *
 *  Created on: Feb 3, 2011
 *      Author: heber
 */

#ifndef PSI3PARSER_PARAMETERS_HPP_
#define PSI3PARSER_PARAMETERS_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <list>
#include <map>
#include <vector>

#include "CodePatterns/Clone.hpp"

#include "Parser/FormatParser_Parameters.hpp"

#include "Parameters/Parameter.hpp"
#include "Parameters/Value.hpp"


class Psi3Parser;

class Psi3Parser_Parameters : public FormatParser_Parameters
{
  // Psi3Parser should be friend to access params directly for types.
  friend class Psi3Parser;
  // ParserPsi3UnitTest needs to be friend to check types contained in params.
  friend class ParserPsi3UnitTest;

public:
  /** Constructor of Psi3Parser_Parameters.
   *
   */
  Psi3Parser_Parameters();

  /** Destructor of Psi3Parser_Parameters.
   *
   */
  virtual ~Psi3Parser_Parameters();

  /** Enumeration of all known Parameters to allow placing them in vectors, maps.
   *
   */
  enum Parameters {
    labelParam,       //!< labelParam, name of the computation appears in output file
    jobtypeParam,     //!< jobtypeParam, type of job to perform (single point, ...)
    wavefunctionParam,//!< wfnParam, wavefunction type
    maxiterParam,     //!< maxiterParam, maximum number of iterations
    referenceParam,   //!< referenceParam, type of self-consistent field calculation
    basisParam,       //!< basisParam, name of the basis
    freeze_coreParam, //!< freeze_coreParam, whether to freeze core orbitals automatically or not
    unitsParam,       //!< unitsParam, unit of length
    dertypeParam,     //!< dertypeParam, type of derivative to use
    originParam,      //!< originParam, origin of the system
    multiplicityParam,//!< multipParam, multiplicity of the molecule (2S+1) in case of unrestricted calculations
    chargeParam,      //!< chargeParam, charge of the molecule
    soccParam,        //!< soccParam, singly occupied orbitals in each irreducible representation
    doccParam,        //!< doccParam, doubly occupied orbitals in each irreducible representation
    subgroupParam,    //!< subgroupParam, subgroup symmetry the system fulfulls
    unique_axisParam, //!< unique_axisParam, unique_axis in case of degeneracy for the given subgroup
    unknownParam};    //!< unknownParam, designates an unknown parameter

  /** Enumeration of all known job typs.
   *
   */
  enum Jobtype {
    SP,         //!< single-point energy calculation
    OPT,        //!< geometry optimization
    DISP,       //!< different displaced geometries
    FREQ,       //!< frequency calculation
    SYMM_FREQ,  //!< frequency calculation for symmetric vibrational modes only
    DBOC,       //!< Diagonal Born-Oppenheimer correction energy calculation
    RESPONSE,   //!< response properties calculation
    unknownJobtype //!< designates an unknown job type
  };

  /** Enumeration of all known wavefunction.
   *
   */
  enum Wavefunction {
    SCF,        //!< Self-Consistent Field
    MP2,        //!< Moeller-Plesset perturbation theory to second order
    MP2R12,        //!< Moeller-Plesset perturbation theory to second order with r12 functional
    CIS,        //!< Complete Inverse Space
    DETCI,        //!< Determinatal Configuration Interaction
    CASSCF,        //!< Complete Active Space Self-Consistent Field
    RASSCF,        //!< Restriced Active Space Self-Consistent Field
    CCSD,        //!< Coupled Cluster with Singles and Doubles
    CCSD_T,        //!< Coupled Cluster with Singles, Doubles, and Triples
    BCCD,        //!< some Coupled Cluster with Singles and Doubles
    BCCD_T,        //!< some Coupled Cluster with Singles, Doubles, and Triples
    EOM_CCSD,        //!< some other Coupled Cluster with Singles and Doubles
    ZAPTN,        //!< unknown type?
    unknownWavefunction //!< designates an unknown wavefunction
  };

  /** Enumeration of all known reference.
   *
   */
  enum Reference {
    RHF,        //!< Restricted Hartree-Fock equations
    ROHF,       //!< Restricted Open Shell Hartree-Fock equations
    UHF,        //!< Unrestricted Hartree-Fock equations
    TWOCON,     //!< Special Unrestricted Hartree-Fock equations
    unknownReference //!< designates an unknown reference
  };

  /** Enumeration of all known unique_axis values.
   *
   */
  enum UniqueAxis {
    X,  //!< x axis
    Y,  //!< y axis
    Z,  //!< z axis
    unknownUniqueAxis //!< designates an unknown unique axis
  };

  /** Enumeration of all known length units.
   *
   */
  enum Units {
    angstrom,  //!< angstroem units of length
    bohr,  //!< bohr units of length
    unknownUnits //!< designates an unknown unit
  };

  /** Enumeration of all known length dertypes.
   *
   */
  enum DerivativeType {
    NONE,  //!< no derivative calculation
    unknownDerivativeType //!< designates an unknown dertype
  };

  /** Enumeration of all known unique freeze_core values.
   *
   */
  enum FreezeCore {
    YES,        //!< freeze cores automatically
    TRUE,       //!< same as YES
    NO,         //!< don't freeze cores
    FALSE,      //!< same as NO
    SMALL,      //!< for elements Na-Ar
    LARGE,      //!< for elements Na-Ar
    unknownFreezeCore //!< designates an unknown freeze_core
  };

  // enum to string getters
  const std::string getParameter(const enum Parameters param) const;
  void setParameter(const enum Parameters param, const std::string &);
  const std::string &getParameterName(const enum Parameters param) const;
  const std::string &getReferenceName(const enum Reference reference) const;

private:
  //!> vector with all available dertype in same order as enum DerivativeType.
  std::vector<std::string> ValidDerivativeType;

  //!> vector with all available unit in same order as enum Units.
  std::vector<std::string> ValidUnits;

  //!> vector with all available unique_axis in same order as enum UniqueAxis.
  std::vector<std::string> ValidUniqueAxis;

  //!> vector with all available freeze_core in same order as enum FreezeCore.
  std::vector<std::string> ValidFreezeCore;

  //!> vector with all available job types in same order as enum Jobtypes.
  std::vector<std::string> ValidJobtypes;

  //!> vector with all available wavefunction in same order as enum Wavefunction.
  std::vector<std::string> ValidWavefunction;

  //!> vector with all available reference in same order as enum Reference.
  std::vector<std::string> ValidReference;

  bool checkWorldElementsAgainstCurrentBasis() const;

  /** Global initialization in cstor.
   *
   */
  void Init();

//  /** Initializes BasisList.
//   *
//   */
//  void initBasis();

  //!> vector with all parameter names in same order as enum Parameters
  std::vector<std::string> ParamNames;

  //!> typedef for the list of all available basis sets
  typedef std::map<std::string, std::list<std::string> > BasisMapType;

  //!> list of all basis along with their present element parametrization
  BasisMapType BasisList;
};

#endif /* PSI3PARSER_PARAMETERS_HPP_ */
