/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * LoadAction.cpp
 *
 *  Created on: May 8, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"
#include "Descriptors/MoleculeIdDescriptor.hpp"
#include "Parser/FormatParserInterface.hpp"
#include "Parser/FormatParserStorage.hpp"
#include "Parser/FormatParser_Parameters.hpp"
#include "molecule.hpp"
#include "MoleculeListClass.hpp"
#include "World.hpp"

#include <iostream>

#include <boost/filesystem/fstream.hpp>

#include "LoadAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "LoadAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
Action::state_ptr MoleculeLoadAction::performCall() {
  // obtain information
  getParametersfromValueStorage();

  // parsing file if present
  if (!boost::filesystem::exists(params.filename)) {
    DoLog(1) && (Log() << Verbose(1) << "Specified input file " << params.filename << " not found." << endl);
    return Action::failure;
  } else {
    DoLog(1) && (Log() << Verbose(1) << "Specified input file found, parsing ... ");

    // extract suffix
    std::string FilenameSuffix;
    std::string FilenamePrefix;
    if (params.filename.has_filename()) {
      // get suffix
#if BOOST_VERSION >= 104600
      FilenameSuffix = params.filename.extension().string().substr(1); // remove the prefixed "."
      FilenamePrefix = params.filename.stem().string();
#else
      FilenameSuffix = params.filename.extension().substr(1); // remove the prefixed "."
      FilenamePrefix = params.filename.stem();
#endif
    } else {
      DoeLog(1) && (eLog() << Verbose(1) << "Input file does not have a suffix, cannot recognize format." << endl);
      return Action::failure;
    }

    // get undo state for parser
    enum ParserTypes type = FormatParserStorage::getInstance().getTypeFromSuffix(FilenameSuffix);
    ASSERT(type != ParserTypes_end,
        "MoleculeLoadAction::performCall() - unknown file suffix "+FilenameSuffix+".");
    FormatParser_Parameters *ParserParams = FormatParserStorage::getInstance().get(type).parameters;
    if (ParserParams != NULL)
      ParserParams = ParserParams->clone();
    else
      ParserParams = NULL;

    // parse the file
    boost::filesystem::ifstream input;
    input.open(params.filename);
    FormatParserStorage::getInstance().load(input, FilenameSuffix);
    input.close();

    // set file name of last molecule
    MoleculeList::const_reverse_iterator iter = World::getInstance().getMolecules()->ListOfMolecules.rbegin();
    (*iter)->SetNameFromFilename(FilenamePrefix.c_str());
    DoLog(0) && (Log() << Verbose(0) << "Chemical formula is " << (*iter)->getFormula() << std::endl);

    return Action::state_ptr(
        new MoleculeLoadState((*iter)->getId(),FilenamePrefix,FilenameSuffix,boost::shared_ptr<FormatParser_Parameters>(ParserParams),params)
    );
  }
}

Action::state_ptr MoleculeLoadAction::performUndo(Action::state_ptr _state) {
  MoleculeLoadState *state = assert_cast<MoleculeLoadState*>(_state.get());

  // remove loaded molecule and its atoms
  molecule *mol = World::getInstance().getMolecule(MoleculeById(state->molId));
  for(molecule::iterator iter = mol->begin(); !mol->empty(); iter = mol->begin())
    World::getInstance().destroyAtom(*iter);
  //World::getInstance().destroyMolecule(mol);

  // undo changes to FormatParser
  enum ParserTypes type = FormatParserStorage::getInstance().getTypeFromSuffix(state->FilenameSuffix);
  FormatParser_Parameters *ParserParams = FormatParserStorage::getInstance().get(type).parameters;
  if (ParserParams != NULL)
    ParserParams->makeClone(*state->ParserParameters);

  return Action::state_ptr(_state);
}

Action::state_ptr MoleculeLoadAction::performRedo(Action::state_ptr _state){
  MoleculeLoadState *state = assert_cast<MoleculeLoadState*>(_state.get());

  // get undo state for parser
  enum ParserTypes type = FormatParserStorage::getInstance().getTypeFromSuffix(state->FilenameSuffix);
  FormatParser_Parameters *ParserParams = FormatParserStorage::getInstance().get(type).parameters;
  if (ParserParams != NULL)
    ParserParams = ParserParams->clone();
  else
    ParserParams = NULL;

  // parse the file
  boost::filesystem::ifstream input;
  input.open(state->params.filename);
  FormatParserStorage::getInstance().load(input, state->FilenameSuffix);
  input.close();

  // set file name of last molecule
  MoleculeList::const_reverse_iterator iter = World::getInstance().getMolecules()->ListOfMolecules.rbegin();
  (*iter)->SetNameFromFilename(state->FilenamePrefix.c_str());
  (*iter)->setId(state->molId);
  DoLog(0) && (Log() << Verbose(0) << "Chemical formula is " << (*iter)->getFormula() << std::endl);

  return Action::state_ptr(
      new MoleculeLoadState((*iter)->getId(),state->FilenamePrefix,state->FilenameSuffix,boost::shared_ptr<FormatParser_Parameters>(ParserParams),params)
  );
}

bool MoleculeLoadAction::canUndo() {
  return true;
}

bool MoleculeLoadAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
