/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * FormatParser_Parameters.cpp
 *
 *  Created on: Sep 30, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include <iostream>
#include <boost/algorithm/string.hpp>
#include <boost/tokenizer.hpp>
#include <sstream>
#include <string>

#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"

#include "Parameters/Parameter.hpp"
#include "FormatParser_Parameters.hpp"

/** Constructor of class FormatParser_Parameters.
 *
 * \note we make sure that storage is always present
 *
 */
FormatParser_Parameters::FormatParser_Parameters() :
  storage(new ParameterStorage)
{}

/** Copy constructor of class FormatParser_Parameters.
 *
 * @param _parameters instance to copy from
 */
FormatParser_Parameters::FormatParser_Parameters(const FormatParser_Parameters &_parameters) :
  storage(new ParameterStorage(*_parameters.storage))
{}


/** Destructor of class FormatParser_Parameters.
 *
 */
FormatParser_Parameters::~FormatParser_Parameters()
{
  delete storage;
}

/** Implements the Parameter::clone() function.
 *
 * @return another instance with an identical copy of each parameter in \a storage
 */
FormatParser_Parameters* FormatParser_Parameters::clone() const
{
  return (new FormatParser_Parameters(*this));
}

/** This makes this instance a clone of \a _instance.
 *
 * \note This is basically the other way round to clone().
 *
 * @param _instance instance to clone from
 */
void FormatParser_Parameters::makeClone(const FormatParser_Parameters &_instance)
{
  // we simply remove storage
  delete storage;
  // and clone the one from _instance
  storage = new ParameterStorage(*_instance.storage);
}

/** Adds a parameter to \a storage.
 *
 * This just eases use, saves some typing and is more clear as to what is done.
 *
 * @param instance parameter to add
 */
void FormatParser_Parameters::appendParameter(Parameter *instance)
{
  storage->registerInstance(instance);
}

/** Checks for presence of a parameter in \a storage.
 *
 * This just eases use, saves some typing and is more clear as to what is done.
 *
 * @return true - parameter by this \a _name is present in storage, false - else
 */
bool FormatParser_Parameters::haveParameter(const std::string &_name) const
{
  return storage->isPresentByName(_name);
}

/** Gets a parameter from \a storage.
 *
 * This just eases use, saves some typing and is more clear as to what is done.
 *
 * @return pointer to instance with this \a _name
 */
Parameter *FormatParser_Parameters::getParameter(const std::string &_name) const
{
  return storage->getByName(_name);
}

/** Output operator for the contents of FormatParser_Parameters::params.
 *
 * @param ost output stream
 * @param params reference to FormatParser_Parameters containing params.
 * @return reference to output stream for concatenation
 */
std::ostream & operator << (std::ostream& ost, const FormatParser_Parameters &params)
{
  // this is ugly, but with boost::any to safeguard const-ness is plain impossible
  std::ostringstream output;
  ASSERT(params.storage != NULL,
      "operator<<(FormatParser_Parameters) - storage is NULL.");
  for (ParameterStorage::const_iterator iter = params.storage->getBeginIter();
      iter != params.storage->getEndIter();
      ++iter)
    if (!iter->second->get().empty())
      output << iter->first <<  "=" << iter->second->get() << ";";
  ost << output.str();
  return ost;
}

/** Input operator for a list of parameters to place into \a params.
 *
 * @param ist input stream
 * @param params parameters to parse into
 * @return input stream for concatenation
 */
std::istream & operator >> (std::istream& ist, FormatParser_Parameters &params)
{
  typedef boost::tokenizer<boost::char_separator<char> >
      tokenizer;
  boost::char_separator<char> semicolonsep(";");
  boost::char_separator<char> equalitysep("=");
  boost::char_separator<char> ticksep("\"");
  std::string line;
  std::getline( ist, line );
  //DoLog(0) && (Log() << Verbose(0) << "INFO: full line of parameters is '" << line << "'" << std::endl);
  tokenizer tokens(line, semicolonsep);
  ASSERT(tokens.begin() != tokens.end(),
      "operator<< on FormatParser_Parameters - empty string, need at least ';' in line "+line+"!");
  for (tokenizer::iterator tok_iter = tokens.begin();
        tok_iter != tokens.end(); ++tok_iter) {
    //LOG(2, "INFO: (key,value) pair is: " << *tok_iter << ".");
    tokenizer paramtokens(*tok_iter, equalitysep);
    if (paramtokens.begin() != paramtokens.end()) {
      tokenizer::iterator tok_paramiter = paramtokens.begin();
      ASSERT(tok_paramiter != paramtokens.end(),
          "operator<< on FormatParser_Parameters - missing value before ' =' in token "+*tok_iter+"!");
      tokenizer::iterator tok_valueiter = tok_paramiter;
      tokenizer::iterator tok_checkiter = ++tok_valueiter;
      ASSERT(tok_valueiter != paramtokens.end(),
          "operator<< on FormatParser_Parameters - missing value after ' =' in token "+*tok_iter+"!");
      ++tok_checkiter;
      ASSERT(tok_checkiter == paramtokens.end(),
          "operator<< on FormatParser_Parameters - still more tokens after ' =' in token "+*tok_iter+":"
          +*tok_checkiter+"!");
      LOG(3, "INFO: key is '" << *tok_paramiter << "', value is '" << *tok_valueiter << "'.");
      // TODO: throw exception instead of ASSERT
      std::string key(*tok_paramiter);
      std::string value(*tok_valueiter);
      boost::trim(key);
      boost::trim(value);
      tokenizer ticklesstokens(value, ticksep);
      ASSERT(ticklesstokens.begin() != ticklesstokens.end(),
          "operator<< on FormatParser_Parameters - no tokens present after removing ticks in token "+*tok_valueiter+"!");
      std::stringstream valuestream(*(ticklesstokens.begin()));
      //LOG(2, "INFO: Token pair is " << key << "," << valuestream.str());

      // TODO: throw exception instead of DoeLog()
      ASSERT(params.haveParameter(key),
          "operator >> on FormatParser_Parameters - unknown parameter name '"
          +key+"' with value "+valuestream.str()+"!");
      if (params.haveParameter(key)) {
        Parameter *instance = params.getParameter(key);
        instance->set(valuestream.str());
      }
    } else {
      ist.setstate(std::ios::eofbit);
    }
  }
  return ist;
}
