/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * QtWorldView.cpp
 *
 *  Created on: Jan 21, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Views/Qt4/QtWorldView.hpp"

#include <QMetaMethod>

#include <iostream>

#include "CodePatterns/MemDebug.hpp"

#include "Atom/atom.hpp"
#include "Formula.hpp"
#include "molecule.hpp"
#include "MoleculeListClass.hpp"
#include "Actions/SelectionAction/Molecules/MoleculeByIdAction.hpp"
#include "Actions/SelectionAction/Molecules/NotMoleculeByIdAction.hpp"

using namespace std;

// maybe this should go with the definition of molecules

// some attributes need to be easier to find for molecules
// these attributes are skiped so far
const int QtWorldView::COLUMNCOUNT = COLUMNTYPES_MAX;
const char *QtWorldView::COLUMNNAMES[QtWorldView::COLUMNCOUNT]={"Name","Atoms","Formula","Occurrence"/*,"Size"*/};

QtWorldView::QtWorldView(QWidget * _parent) :
  QTreeWidget (_parent),
  Observer("QtWorldView")
{
  setColumnCount(COLUMNCOUNT);
  setSelectionMode(QAbstractItemView::MultiSelection);

  QStringList header;
  for(int i=0; i<COLUMNCOUNT;++i)
    header << COLUMNNAMES[i];
  setHeaderLabels(header);

  World::getInstance().signOn(this);//, World::MoleculeInserted);
  //World::getInstance().signOn(this, World::MoleculeRemoved);


  //for (int i=0;i<metaObject()->methodCount();i++)
  //  std::cout << metaObject()->method(i).signature() << "   -   " << metaObject()->method(i).methodType() << "\n";



  dirty = true;
  clearing = false;
  selecting = false;
  refill();

  //connect(this,SIGNAL(cellChanged(int,int)),this,SLOT(moleculeChanged(int,int)));
  connect(selectionModel(),SIGNAL(selectionChanged(QItemSelection, QItemSelection)),this,SLOT(rowsSelected(QItemSelection, QItemSelection)));

}

QtWorldView::~QtWorldView()
{
  World::getInstance().signOff(this);//, World::MoleculeInserted);
  //World::getInstance().signOff(this, World::MoleculeRemoved);
}

void QtWorldView::update(Observable *publisher) {

  if (selecting)
    return;

  dirty = true;

  // force an update from Qt...
  clearing = true;
  clear();
  clearing = false;
}

void QtWorldView::refill() {
  const std::vector<molecule*> &molecules = World::getInstance().getAllMolecules();

  clear();

  // list of (unique) formulas in the world
  std::vector<Formula> formula;

  for (std::vector<molecule*>::const_iterator iter = molecules.begin();
      iter != molecules.end();
      iter++) {

    // find group if already in list
    QTreeWidgetItem *groupItem = NULL;
    for (unsigned int j=0;j<formula.size();j++)
      if ((*iter)->getFormula() == formula[j]){
        groupItem = topLevelItem(j);
        break;
      }

    // new molecule type -> create new group
    if (!groupItem){
      formula.push_back((*iter)->getFormula());
      groupItem = new QTreeWidgetItem(this);
      groupItem->setText(0, QString((*iter)->getName().c_str()));
      groupItem->setText(1, QString::number((*iter)->getAtomCount()));
      groupItem->setText(2, QString((*iter)->getFormula().toString().c_str()));
      groupItem->setText(3, "0");
      groupItem->setData(0, Qt::UserRole, QVariant(-1));
    }

    // add molecule
    QTreeWidgetItem *molItem = new QTreeWidgetItem(groupItem);
    molItem->setText(0, QString((*iter)->getName().c_str()));
    molItem->setText(1, QString::number((*iter)->getAtomCount()));
    molItem->setText(2, QString((*iter)->getFormula().toString().c_str()));
    const int index = (*iter)->getId();
    molItem->setData(0, Qt::UserRole, QVariant(index));
    molItem->setSelected(World::getInstance().isSelected(*iter));


    // increase group occurrence
    int count = groupItem->text(3).toInt() + 1;
    groupItem->setText(3, QString::number(count));
  }
  dirty = false;
}

void QtWorldView::paintEvent(QPaintEvent * event)
{
  if (dirty)
    refill();
  QTreeWidget::paintEvent(event);
}

void QtWorldView::subjectKilled(Observable *publisher) {
}

void QtWorldView::moleculeChanged() {
  /*int idx = verticalHeaderItem(row)->data(Qt::UserRole).toInt();
  molecule *mol = molecules->ReturnIndex(idx);
  string cellValue = item(row,NAME)->text().toStdString();
  if(mol->getName() != cellValue && cellValue !="") {
    mol->setName(cellValue);
  }
  else if(cellValue==""){
    item(row,NAME)->setText(QString(mol->getName().c_str()));
  }*/
}

void QtWorldView::rowsSelected(const QItemSelection & selected, const QItemSelection & deselected){

  if (clearing)
    return;
  if (selecting)
    return;
  selecting = true;

  // Select all molecules which belong to newly selected rows.
  QModelIndex index;
  QModelIndexList items = selected.indexes();
  foreach (index, items)
    if (index.column() == 0){
      int mol_id = model()->data(index, Qt::UserRole).toInt();
      if (mol_id < 0)
        continue;
      //std::cout << "select molecule" << std::endl;
      MoleCuilder::SelectionMoleculeById(mol_id);
    }

  // Unselect all molecules which belong to newly unselected rows.
  items = deselected.indexes();
  foreach (index, items)
    if (index.column() == 0){
      int mol_id = model()->data(index, Qt::UserRole).toInt();
      if (mol_id < 0)
        continue;
      //std::cout << "unselect molecule" << std::endl;
      MoleCuilder::SelectionNotMoleculeById(mol_id);
    }

  selecting = false;
}
