/*
 * UIFactory.hpp
 *
 *  Created on: Jan 5, 2010
 *      Author: crueger
 */

#ifndef UIFACTORY_HPP_
#define UIFACTORY_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif


class MainWindow;
class Dialog;
class DialogDescription;

#include "CodePatterns/Singleton.hpp"

#include <string>
#include <map>
#include <boost/smart_ptr.hpp>

/**
 * Abstract Factory to create any kind of User interface object needed by the programm.
 *
 * The factory can be created and has to be set to a certain type upon creation. It will then
 * only create UIelements of that certain type, so that all UIElements match. This way different
 * UIs can be handled in a concise abstract way.
 *
 * The main functionality is the MainWindow and the Dialog.
 *
 * MainWindow basically is the framework if the UI. The MainWindow has a run function which is
 * called MainWindow::Display(), which will listen for user input and react to it via actions for
 * as long as the user desires or tells the MainWindow to quit.
 *
 * Within the MainWindow Dialog classes may be instantiated which ask the user for input to
 * certain actions he wants to perform.
 *
 */
class UIFactory : public Singleton<UIFactory,false>
{
  friend class Singleton<UIFactory,false>;
public:

  /**
   * Produce some kind of main window, of whichever type was chosen when the factory was created
   */
  virtual MainWindow* makeMainWindow()=0;

  /**
   * Produce a User Interaction Dialog, that can query values from the User.
   * Again the type is determined upon factory creation.
   */
  virtual Dialog* makeDialog()=0;

protected:
  UIFactory();
  virtual ~UIFactory();

public:
  struct factoryDescription {
    factoryDescription(std::string _name);
    virtual ~factoryDescription();

    const std::string name;
    // yes this method really is a factory factory, to allow insertion of
    // arbitrary factories
    virtual UIFactory* makeFactory()=0;
  };
  /**
   * create a Factory of a certain type. From that moment on only those UIElements can be produced by the factory
   */
  static void makeUserInterface(std::string type);
  static void registerFactory(factoryDescription *factoryDesc);
  static bool isFactoryPresent() { return !factories.empty(); }
  virtual std::string getUIName(){  return "none";  }
protected:
private:
  static std::map<std::string,boost::shared_ptr<factoryDescription> > factories;
};

#endif /* UIFACTORY_HPP_ */
