/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * QtInfoBox.cpp
 *
 *  Created on: Mar 4, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Views/Qt4/QtInfoBox.hpp"

#include <iostream>
#include <QAbstractItemView>

#include "CodePatterns/MemDebug.hpp"

#include "molecule.hpp"
#include "Element/element.hpp"

using namespace std;

/***************** Basic structure for tab layout ***********/

QtInfoBox::QtInfoBox() :
    QTabWidget(),
    curAtom(NULL), nextAtom(NULL),
    curMolecule(NULL), nextMolecule(NULL),
    page_mol(NULL), page_atom(NULL)
{
  timer = new QTimer(this);
  timer->setSingleShot(true);

  setMinimumWidth(200);
  setMinimumHeight(220);
  currentPage = 0;

  connect(timer, SIGNAL(timeout()), this, SLOT(timerTimeout()));
}

QtInfoBox::~QtInfoBox()
{
  clearTabs();
}

void QtInfoBox::atomHover(const atom &_atom)
{
  nextAtom = &_atom;
  timer->start(500);
}

void QtInfoBox::moleculeHover(const molecule &_mol)
{
  nextMolecule = &_mol;
  timer->start(500);
}

void QtInfoBox::timerTimeout()
{
  if (nextAtom)
    showAtom(nextAtom);
  if (nextMolecule)
    showMolecule(nextMolecule);
}

void QtInfoBox::clearTabs()
{
  if (page_atom){
    //removeTab(indexOf(page_atom));
    delete(page_atom);
    page_atom = NULL;
  }
  if (page_mol){
    //removeTab(indexOf(page_mol));
    delete(page_mol);
    page_mol = NULL;
  }
}

void QtInfoBox::showAtom(const atom *_atom)
{
  currentPage = currentIndex();

  // Remove old tabs.
  clearTabs();

  curAtom = _atom;
  nextAtom = NULL;
  nextMolecule = NULL;

  // Show new tabs.
  if (curAtom){
    page_atom = new QtAtomInfoPage(curAtom, this);
    addTab(page_atom, "Atom");
    connect(page_atom, SIGNAL(atomKilled()), this, SLOT(clearTabs()));

    if (curAtom->getMolecule()){
      page_mol = new QtMoleculeInfoPage(curAtom->getMolecule(), this);
      addTab(page_mol, "Molecule");
      connect(page_mol, SIGNAL(moleculeKilled()), this, SLOT(clearTabs()));

      if (currentPage > 0)
        setCurrentIndex(currentPage);
    }
  }
}

void QtInfoBox::showMolecule(const molecule *_mol)
{
  currentPage = currentIndex();

  // Remove old tabs.
  clearTabs();

  curMolecule = _mol;
  nextAtom = NULL;
  nextMolecule = NULL;

  // Show new tabs.
  if (curMolecule){
    page_mol = new QtMoleculeInfoPage(curMolecule, this);
    addTab(page_mol, "Molecule");
    connect(page_mol, SIGNAL(moleculeKilled()), this, SLOT(clearTabs()));

    if (currentPage > 0)
      setCurrentIndex(currentPage);
  }
}

/************************ Tab for single Atoms ********************/

static void addInfo(QTreeWidget *info, const QString &key, const QString &value)
{
  QTreeWidgetItem *treeItem = new QTreeWidgetItem(info);
  treeItem->setText(0, key);
  treeItem->setText(1, value);
}

QtAtomInfoPage::QtAtomInfoPage(const atom *_atom, QWidget *parent) :
  QTreeWidget(parent),
  Observer("QTAtomPage"),
  atomRef(_atom)
{
  atomRef->signOn(this);

  setColumnCount(2);
  QStringList header;
  header << "data";
  header << "value";
  setHeaderLabels(header);

  addInfo(this, "Name", QString(atomRef->getName().c_str()));
  addInfo(this, "Element", QString(atomRef->getElement().getName().c_str()));
  addInfo(this, "Mass", QString("%1").arg(atomRef->getMass()));
  addInfo(this, "Charge", QString("%1").arg(atomRef->getCharge()));
  addInfo(this, "Bonds", QString("%1").arg(atomRef->getListOfBonds().size()));
  addInfo(this, "Position x", QString(toString(atomRef->getPosition()[0]).c_str()));
  addInfo(this, "Position y", QString(toString(atomRef->getPosition()[1]).c_str()));
  addInfo(this, "Position z", QString(toString(atomRef->getPosition()[2]).c_str()));
}

QtAtomInfoPage::~QtAtomInfoPage()
{
  if (atomRef)
    atomRef->signOff(this);
}

void QtAtomInfoPage::update(Observable *subject){
  /*if(name != atomRef->name){
    name = atomRef->name;
    emit nameChanged(this,name);
  }*/
}

void QtAtomInfoPage::subjectKilled(Observable *subject){
  atomRef = NULL;
  emit atomKilled();
}

/************************ Tab for single Molecules *****************/

QtMoleculeInfoPage::QtMoleculeInfoPage(const molecule *_mol, QWidget *parent) :
    QTreeWidget(parent),
    Observer("QTMoleculePage"),
    mol(_mol)
{
  mol->signOn(this);

  setColumnCount(2);
  QStringList header;
  header << "data";
  header << "value";
  setHeaderLabels(header);

  addInfo(this, "Name", QString(mol->getName().c_str()));
  addInfo(this, "Formula", QString(mol->getFormula().toString().c_str()));
  addInfo(this, "Atoms", QString("%1").arg(mol->getAtomCount()));
  addInfo(this, "NonHydrogens", QString("%1").arg(mol->getNoNonHydrogen()));
  addInfo(this, "Bonds", QString("%1").arg(mol->getBondCount()));
  const Vector molCenter = mol->DetermineCenterOfAll();
  addInfo(this, "Center x", QString("%1").arg(molCenter[0]));
  addInfo(this, "Center y", QString("%1").arg(molCenter[1]));
  addInfo(this, "Center z", QString("%1").arg(molCenter[2]));
}

QtMoleculeInfoPage::~QtMoleculeInfoPage(){
  if (mol)
    mol->signOff(this);
}

void QtMoleculeInfoPage::update(Observable *subject){  std::cout << "tab mol update\n";
}

void QtMoleculeInfoPage::subjectKilled(Observable *subject){
  mol = NULL;
  emit moleculeKilled();
}
