/*
 * Histogram.hpp
 *
 *  Created on: Jul 26, 2012
 *      Author: heber
 */

#ifndef HISTOGRAM_HPP_
#define HISTOGRAM_HPP_


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iosfwd>
#include <map>
#include <vector>

class HistogramTest;

/** This class generates a histogram from a given vector of sampled values.
 *
 * Most importantly, it also contains operator+=() and operator-=() to perform
 * sum on the \b histograms.
 *
 * This is to be used with the OrthogonalSummation which requires these specific
 * operator implementations.
 *
 */
class Histogram
{
  //!> grant unit test access to private parts
  friend class HistogramTest;
public:
  //!> named type for a vector of sample values
  typedef std::vector<double> samples_t;
  //!> named type for the start of a bin
  typedef double BinLowerEnd;
  //!> named type for the count of a bin, may be fractional
  typedef double BinWeight;
  //!> named type for the pair identifying a bin by its lower end and count
  typedef std::pair< BinLowerEnd, BinWeight> Bin_t;
private:
  //!> named type for a vector of bins
  typedef std::map< BinLowerEnd, BinWeight > Bins_t;
public:
  /** Constructor for class Histogram.
   *
   * @param samples samples to put in the histogram.
   * @param CountBins number of bins (if negative, we choose number by statistical means)
   */
  Histogram(const samples_t &samples, const int _CountBins=-1);

  /** Adding another histogram onto this one.
   *
   * @param other other histogram
   * @return ref to this instance
   */
  Histogram& operator+=(const Histogram &other);

  /** Subtracting another histogram from this one.
   *
   * @param other other histogram
   * @return ref to this instance
   */
  Histogram& operator-=(const Histogram &other);

  /** States whether each bin of this histogram has count of zero.
   *
   * @return true - all bins are zero, false - else
   */
  bool isEmpty() const;

private:
  /** Returns the iterator to the bin representing the lower end into which the \a value fits.
   *
   * @param _value value to fit.
   * @return iterator to bin or to bins.end() if not matching
   */
  Bins_t::iterator getLowerEndBin(const double _value);

  /** Returns the iterator to the bin representing the upper end into which the \a value fits.
   *
   * @param _value value to fit.
   * @return iterator to bin or to bins.end() if not matching
   */
  Bins_t::iterator getHigherEndBin(const double _value);

private:
  //!> vector of bins containing the histogram
  Bins_t bins;
  //!> width of bin
  double binwidth;
  //!> number of bins
  int CountBins;
};

/** Function to print an arbitrary pair to ostream.
 *
 * @param ost output stream
 * @param elem element to print
 * @return ref to given ostream for concatenation
 */
std::ostream & operator<<(std::ostream &ost, const Histogram::Bin_t &elem);


#endif /* HISTOGRAM_HPP_ */
