/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * TxMenu.cpp
 *
 *  Created on: Dec 10, 2009
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include <boost/bind.hpp>
#include <algorithm>
#include <iostream>
#include <cmath>
#include "cleanUp.hpp"
#include "Menu/TextMenu/TxMenu.hpp"
#include "Menu/TextMenu/MenuItem.hpp"


/** Constructor for class TxMenu.
 *
 * produce a text menu with a given title.
 * The text will later be displayed using the stream passed to the constructor.
 * \param &_outputter output stream to use for displaying the text
 * \param _title title of this menu
 * \param _spacer key to separate trigger key from descriptive text shown
 * \param _length maximum length of the descriptive text
 */
TxMenu::TxMenu(std::ostream& _outputter, const std::string _title, char _spacer,int _length) :
  defaultItem(0),
  outputter(_outputter),
  title(_title),
  spacer(_spacer),
  length(_length),
  quit(false)
{
}

/** Destructor for class TxMenu.
 *
 */
TxMenu::~TxMenu()
{
  for(std::list<MenuItem*>::iterator it=items.begin(); !items.empty(); it=items.begin()) {
    delete (*it);
    items.erase(it);
  }
}

/** Adds an MenuItem to the internal list.
 * \param *item item to add
 */
void TxMenu::addItem(MenuItem* item) {
  items.push_back(item);
}

/** Removes an MenuItem to the internal list.
 * \param *item item to remove
 */
void TxMenu::removeItem(MenuItem* item) {
  items.remove(item);
}

/** Function to quit this TxMenu.
 */
void TxMenu::doQuit(){
  quit = true;
}

/** Return the current state of quitting.
 * \return quit boolean
 */
bool TxMenu::hasQuit(){
  return quit;
}

/** Display in a formatted manner a given entry of this menu.
 * \param *entry MenuItem to show
 */
void TxMenu::showEntry(MenuItem* entry){
  if(entry->isActive()==false){
    outputter << "(";
  }
  outputter << entry->formatEntry();
  if(entry->isActive()==false){
    outputter << ")";
  }
  outputter << "\n";
}

/** Display this menu.
 *
 */
void TxMenu::display() {
  char choice;
  bool somethingChosen = false;
  quit = false;
  do {
    int pre = floor((length - title.length()) /2.0);
    int post = ceil((length - title.length()) /2.0);
    for(int i=0;i<pre;i++)
      outputter << spacer;
    outputter << title;
    for(int i=0;i<post;i++)
          outputter << spacer;
    outputter << '\n';
    for_each(items.begin(), items.end(), boost::bind(&TxMenu::showEntry,this,_1));
    outputter.flush();

    std::cin >> choice;

    std::list<MenuItem*>::iterator iter;
    for(iter = items.begin(); iter!=items.end();iter++){
      if((*iter)->isActive()){
        somethingChosen |= (*iter)->checkTrigger(choice);
      }
    }
    // see if something was chosen and call default Item if not
    if(!somethingChosen) {
      if(defaultItem){
        defaultItem->doTrigger();
      }
      else{
        outputter << "Invalid Choice!" <<  std::endl;
      }
    } 
#ifdef HAVE_ACTION_THREAD
    // wait for actions to finish
    waitQueue();
#endif
  }while (!hasQuit());
}

/**  Return the internally stored title of the menu.
 * \return title string
 */
std::string TxMenu::getTitle(){
  return title;
}

/**  Return the internally stored outputter of the menu.
 * \return output stream reference
 */
std::ostream& TxMenu::getOutputter()
{
  return outputter;
}

/** Add a default item to the menu.
 * \param *_defaultItem MenuItem to act as default item.
 */
void TxMenu::addDefault(MenuItem* _defaultItem) {
  defaultItem = _defaultItem;
}

