/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * Copyright (C)  2013 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * AtomObserverUnitTest.cpp
 *
 *  Created on: Dec 19, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "AtomObserverUnitTest.hpp"

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>
#include <iostream>

#include "CodePatterns/Log.hpp"

#include "World.hpp"
#include "Atom/atom.hpp"
#include "Atom/AtomObserver.hpp"
#include "Element/element.hpp"
#include "Element/periodentafel.hpp"

#include "stubs/ObserverStub.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/
// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( AtomObserverTest );

// set up and tear down
void AtomObserverTest::setUp(){
  World::getInstance();
  AtomObserver::getInstance();

  setVerbosity(3);

  // observer checks that we don't receive general updates ...
  observer = new UpdateCountObserver;
  // but only specific notifications in observer1 and observer2
  observer1 = new NotificationObserver(
      AtomObserver::getInstance().getChannel(AtomObservable::PositionChanged)
      );
  observer2 = new NotificationObserver(
      AtomObserver::getInstance().getChannel(AtomObservable::ElementChanged)
      );

  AtomObserver::getInstance().signOn(observer);
  AtomObserver::getInstance().signOn(observer1, AtomObservable::PositionChanged);
  AtomObserver::getInstance().signOn(observer2, AtomObservable::ElementChanged);

  for(int i=0;i<ATOM_COUNT;++i){
    atoms[i]= World::getInstance().createAtom();
    atomIds[i]= atoms[i]->getId();
  }
}

void AtomObserverTest::tearDown()
{
  AtomObserver::getInstance().signOff(observer);
  AtomObserver::getInstance().signOff(observer1, AtomObservable::PositionChanged);
  AtomObserver::getInstance().signOff(observer2, AtomObservable::ElementChanged);

  delete observer;
  delete observer1;
  delete observer2;

  World::purgeInstance();
  AtomObserver::purgeInstance();
}

/** Unit test on whether adding/removing atoms informs relay and onward.
 *
 */
void AtomObserverTest::AtomAddTest()
{
  // check for zero update
  CPPUNIT_ASSERT_EQUAL( (int)0, observer->updates );
  CPPUNIT_ASSERT_EQUAL( false, observer1->wasNotified );
  CPPUNIT_ASSERT_EQUAL( false, observer2->wasNotified );
}

/** Unit test on whether moving atoms informs relay and onward.
 *
 */
void AtomObserverTest::AtomMoveTest()
{
  // move an atom
  atoms[0]->setPosition(Vector(1.,0.,0.));

  // check for update
  CPPUNIT_ASSERT_EQUAL( true, observer1->wasNotified );
  CPPUNIT_ASSERT_EQUAL( false, observer2->wasNotified );
  CPPUNIT_ASSERT_EQUAL( (int)0, observer->updates );
}

/** Unit test on whether changing atom's element informs relay and onward.
 *
 */
void AtomObserverTest::AtomElementTest()
{
  CPPUNIT_ASSERT( atoms[0]->getType()->getAtomicNumber() != 2 );
  atoms[0]->setType(2);

  // check for update
  CPPUNIT_ASSERT_EQUAL( false, observer1->wasNotified );
  CPPUNIT_ASSERT_EQUAL( true, observer2->wasNotified );
  CPPUNIT_ASSERT_EQUAL( (int)0, observer->updates );
}
