/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * QtDialog.cpp
 *
 *  Created on: Jan 18, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "UIElements/Qt4/QtDialog.hpp"
#include "UIElements/Qt4/Query/QtQuery.hpp"

#include <QtGui/QDialogButtonBox>
#include <Qt/qpushbutton.h>
#include <Qt/qboxlayout.h>

#include "CodePatterns/MemDebug.hpp"


using namespace std;

QtDialog::QtDialog(const std::string &_title) :
    QDialog(0),
    Dialog(_title)
{
  setWindowTitle(QString(_title.c_str()));
  // creating and filling of the Dialog window
  mainLayout = new QVBoxLayout();
  inputLayout = new QVBoxLayout();
  buttonLayout = new QVBoxLayout();
  setLayout(mainLayout);
  mainLayout->addLayout(inputLayout);
  mainLayout->addLayout(buttonLayout);
  buttons = new QDialogButtonBox(QDialogButtonBox::Ok| QDialogButtonBox::Cancel);
  buttonLayout->addWidget(buttons);

  // Disable the ok button until something was entered
  buttons->button(QDialogButtonBox::Ok)->setEnabled(false);

  // connect the buttons to their appropriate slots
  connect(buttons, SIGNAL(accepted()), this, SLOT(accept()));
  connect(buttons, SIGNAL(rejected()), this, SLOT(reject()));
}

QtDialog::~QtDialog()
{
}

bool QtDialog::display(){
  // Button state might have changed by some update that
  // was done during query construction. To make sure
  // the state is correct, we just call update one more time.
  update();
  if(exec()) {
    setAll();
    return true;
  }
  else {
    return false;
  }
}

void QtDialog::update(){
  buttons->button(QDialogButtonBox::Ok)->setEnabled(checkAll());
}

/************************** Query Infrastructure ************************/

void QtDialog::queryEmpty(const std::string title, const std::string description)
{
  registerQuery(new EmptyQtQuery(title,description,inputLayout,this));
}

void QtDialog::queryBoolean(Parameter<bool> &param, const std::string title, const std::string description)
{
  registerQuery(new BooleanQtQuery(param,title,description,inputLayout,this));
}

void QtDialog::queryAtom(Parameter<const atom *> &param, const std::string title, const std::string description)
{
  registerQuery(new AtomQtQuery(param,title,description,inputLayout,this));
}

void QtDialog::queryAtoms(Parameter<std::vector<const atom *> > &param, const std::string title, const std::string description)
{
  registerQuery(new AtomsQtQuery(param,title,description,inputLayout,this));
}

void QtDialog::queryRealSpaceMatrix(Parameter<RealSpaceMatrix> &param, const std::string title, const std::string description)
{
  registerQuery(new RealSpaceMatrixQtQuery(param,title,description,inputLayout,this));
}

void QtDialog::queryInt(Parameter<int> &param, const std::string title, const std::string description)
{
  registerQuery(new IntQtQuery(param,title,description,inputLayout,this));
}

void QtDialog::queryInts(Parameter<std::vector<int> > &param, const std::string title, const std::string description)
{
  registerQuery(new IntsQtQuery(param,title,description,inputLayout,this));
}

void QtDialog::queryUnsignedInt(Parameter<unsigned int> &param, const std::string title, const std::string description)
{
  registerQuery(new UnsignedIntQtQuery(param,title,description,inputLayout,this));
}

void QtDialog::queryUnsignedInts(Parameter<std::vector<unsigned int> > &param, const std::string title, const std::string description)
{
  registerQuery(new UnsignedIntsQtQuery(param,title,description,inputLayout,this));
}

void QtDialog::queryDouble(Parameter<double> &param, const std::string title, const std::string description)
{
  registerQuery(new DoubleQtQuery(param,title,description,inputLayout,this));
}

void QtDialog::queryDoubles(Parameter<std::vector<double> > &param, const std::string title, const std::string description)
{
  registerQuery(new DoublesQtQuery(param,title,description,inputLayout,this));
}

void QtDialog::queryString(Parameter<std::string> &param, const std::string title, const std::string description)
{
  registerQuery(new StringQtQuery(param,title,description,inputLayout,this));
}

void QtDialog::queryStrings(Parameter<std::vector<std::string> > &param, const std::string title, const std::string description)
{
  registerQuery(new StringsQtQuery(param,title,description,inputLayout,this));
}

void QtDialog::queryMolecule(Parameter<const molecule *> &param, const std::string title, const std::string description)
{
  registerQuery(new MoleculeQtQuery(param,title,description,inputLayout,this));
}

void QtDialog::queryMolecules(Parameter<std::vector<const molecule *> > &param, const std::string title, const std::string description)
{
  registerQuery(new MoleculesQtQuery(param,title,description,inputLayout,this));
}

void QtDialog::queryVector(Parameter<Vector> &param, const std::string title, const std::string description)
{
  registerQuery(new VectorQtQuery(param,title,description,inputLayout,this));
}

void QtDialog::queryVectors(Parameter<std::vector<Vector> > &param, const std::string title, const std::string description)
{
  registerQuery(new VectorsQtQuery(param,title,description,inputLayout,this));
}

void QtDialog::queryElement(Parameter<const element *> &param, const std::string title, const std::string description)
{
  registerQuery(new ElementQtQuery(param,title,description,inputLayout,this));
}

void QtDialog::queryElements(Parameter<std::vector<const element *> > &param, const std::string title, const std::string description)
{
  registerQuery(new ElementsQtQuery(param,title,description,inputLayout,this));
}

void QtDialog::queryFile(Parameter<boost::filesystem::path> &param, const std::string title, const std::string description)
{
  registerQuery(new FileQtQuery(param,title,description,inputLayout,this));
}

void QtDialog::queryFiles(Parameter<std::vector< boost::filesystem::path> >&param, const std::string title, const std::string description)
{
  registerQuery(new FilesQtQuery(param,title,description,inputLayout,this));
}

/************************** Query Infrastructure ************************/
/*       ---> shifted to folder Query                                   */
/************************************************************************/

/*************************** Pipe Infrastructure ************************/
/*       ---> shifted to folder Pipe                                    */
/************************************************************************/
