/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * RemoveAction.cpp
 *
 *  Created on: May 9, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "atom.hpp"
#include "AtomicInfo.hpp"
#include "Descriptors/AtomDescriptor.hpp"
#include "CodePatterns/Log.hpp"
#include "molecule.hpp"
#include "CodePatterns/Verbose.hpp"
#include "World.hpp"

#include <iostream>
#include <string>

using namespace std;

#include "Actions/AtomAction/RemoveAction.hpp"

// and construct the stuff
#include "RemoveAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
Action::state_ptr AtomRemoveAction::performCall() {
  // create undo state
  std::vector<AtomicInfo> Walkers;
  for (World::AtomSelectionConstIterator iter = World::getInstance().beginAtomSelection();
      iter != World::getInstance().endAtomSelection();
      ++iter) {
    Walkers.push_back(AtomicInfo(*(iter->second)));
  }
  AtomRemoveState *UndoState = new AtomRemoveState(Walkers, params);

  // remove all selected atoms
  for (World::AtomSelectionConstIterator iter = World::getInstance().beginAtomSelection();
      iter != World::getInstance().endAtomSelection();
      iter = World::getInstance().beginAtomSelection()) {
    DoLog(1) && (Log() << Verbose(1) << "Removing atom " << (iter->second)->getId() << "." << endl);
    World::getInstance().destroyAtom((iter->second));
  }
  return Action::state_ptr(UndoState);
}

Action::state_ptr AtomRemoveAction::performUndo(Action::state_ptr _state) {
  AtomRemoveState *state = assert_cast<AtomRemoveState*>(_state.get());

  size_t i=0;
  for (; i<state->Walkers.size(); ++i) {
    // re-create the atom
    DoLog(1) && (Log() << Verbose(1) << "Re-adding atom " << state->Walkers[i].getId() << "." << endl);
    atom *Walker = World::getInstance().createAtom();
    if (!state->Walkers[i].setAtom(*Walker)) {
      DoeLog(1) && (eLog() << Verbose(1) << "Failed to set id." << endl);
      World::getInstance().destroyAtom(Walker);
      break;
    }
  }
  if (i<state->Walkers.size()) {
    // remove all previous ones, too
    for (size_t j=0;j<i;++j)
      World::getInstance().destroyAtom(state->Walkers[j].getId());
    // and announce the failure of the undo
    return Action::failure;
  }
  return Action::state_ptr(_state);
}

Action::state_ptr AtomRemoveAction::performRedo(Action::state_ptr _state){
  AtomRemoveState *state = assert_cast<AtomRemoveState*>(_state.get());

  // simple remove again all previously added atoms
  for (size_t i=0; i<state->Walkers.size(); ++i) {
    DoLog(1) && (Log() << Verbose(1) << "Re-removing atom " << state->Walkers[i].getId() << "." << endl);
    World::getInstance().destroyAtom(state->Walkers[i].getId());
  }

  return Action::state_ptr(_state);
}

bool AtomRemoveAction::canUndo() {
  return true;
}

bool AtomRemoveAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
