/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * PairCorrelationAction.cpp
 *
 *  Created on: May 9, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

//#include "CodePatterns/MemDebug.hpp"

#include "Analysis/analysis_correlation.hpp"
#include "CodePatterns/Verbose.hpp"
#include "CodePatterns/Log.hpp"
#include "Descriptors/AtomTypeDescriptor.hpp"
#include "Element/element.hpp"
#include "molecule.hpp"
#include "Element/periodentafel.hpp"
#include "LinearAlgebra/RealSpaceMatrix.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "World.hpp"

#include <iostream>
#include <string>

#include "Actions/AnalysisAction/PairCorrelationAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "PairCorrelationAction.def"
#include "Action_impl_pre.hpp"

/** =========== define the function ====================== */
ActionState::ptr AnalysisPairCorrelationAction::performCall() {
  ofstream output;
  ofstream binoutput;
  string type;
  BinPairMap *binmap = NULL;

  // execute action
  output.open(params.outputname.get().string().c_str());
  binoutput.open(params.binoutputname.get().string().c_str());
  PairCorrelationMap *correlationmap = NULL;
  World::ConstAtomComposite atoms_first;
  World::ConstAtomComposite atoms_second;
  if (params.elements.get().size() == 2) {
    std::vector<const element *>::const_iterator elemiter = params.elements.get().begin();
    atoms_first = World::getConstInstance().getAllAtoms(AtomByType(*(elemiter++)));
    if (elemiter != params.elements.get().end()) {
      atoms_second = World::getConstInstance().getAllAtoms(AtomByType(*(elemiter++)));
    } else {
      atoms_second = World::getConstInstance().getSelectedAtoms();
    }
  } else if (params.elements.get().size() == 1) {
    std::vector<const element *>::const_iterator elemiter = params.elements.get().begin();
    atoms_first = World::getConstInstance().getSelectedAtoms();
    atoms_second = World::getConstInstance().getAllAtoms(AtomByType(*(elemiter++)));
  } else {
    STATUS("Either one or two elements must be given.");
    return Action::failure;
  }
  double max_distance = params.BinEnd.get();
  if (params.BinEnd.get() <= 0.) {
    // find max distance within box from diagonal
    const RealSpaceMatrix &M = World::getInstance().getDomain().getM();
    max_distance = (M * Vector(1.,1.,1.)).NormSquared();
  }
  correlationmap = PairCorrelation(atoms_first, atoms_second, max_distance);
  OutputCorrelationMap<PairCorrelationMap>(&output, correlationmap, OutputPairCorrelation_Header, OutputPairCorrelation_Value);
  binmap = BinData( correlationmap, params.BinWidth.get(), params.BinStart.get(), params.BinEnd.get() );
  OutputCorrelationMap<BinPairMap> ( &binoutput, binmap, OutputCorrelation_Header, OutputCorrelation_Value );
  delete binmap;
  delete correlationmap;
  output.close();
  binoutput.close();
  return Action::success;
}

ActionState::ptr AnalysisPairCorrelationAction::performUndo(ActionState::ptr _state) {
  return Action::success;
}

ActionState::ptr AnalysisPairCorrelationAction::performRedo(ActionState::ptr _state){
  return Action::success;
}

bool AnalysisPairCorrelationAction::canUndo() {
  return true;
}

bool AnalysisPairCorrelationAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
