/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * GLMoleculeObject_bond.cpp
 *
 *  Created on: Aug 17, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "GLMoleculeObject_bond.hpp"

#include <Qt3D/qglmaterial.h>
#include <Qt3D/qglscenenode.h>

#include "CodePatterns/MemDebug.hpp"


#include <cmath>

#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"
#include "CodePatterns/Observer/Notification.hpp"
#include "CodePatterns/Observer/ObserverLog.hpp"
#include "Atom/atom.hpp"
#include "Bond/bond.hpp"
#include "Element/element.hpp"
#include "Helpers/defs.hpp"
#include "LinearAlgebra/Line.hpp"
#include "LinearAlgebra/Vector.hpp"

GLMoleculeObject_bond::GLMoleculeObject_bond(QGLSceneNode *mesh[], QObject *parent, const bond::ptr bondref, const enum SideOfBond side) :
  GLMoleculeObject(mesh, parent),
  Observer(std::string("GLMoleculeObject_bond")
      +toString(bondref->leftatom->getId())
      +std::string("-")
      +toString(bondref->rightatom->getId())),
  _bond(*bondref),
  leftatom(bondref->leftatom),
  rightatom(bondref->rightatom),
  leftatomId(bondref->leftatom->getId()),
  rightatomId(bondref->rightatom->getId()),
  BondSide(side)
{
  // sign on as observer (obtain non-const instance before)
  _bond.signOn(this, BondObservable::BondRemoved);
  _bond.signOn(this, BondObservable::DegreeChanged);
  leftatom->signOn(this, AtomObservable::PositionChanged);
  leftatom->signOn(this, AtomObservable::ElementChanged);
  rightatom->signOn(this, AtomObservable::PositionChanged);
  rightatom->signOn(this, AtomObservable::ElementChanged);

  size_t elementno = 0;
  switch (BondSide) {
    case left:
      if (_bond.rightatom->getType() != NULL) {
        elementno = _bond.rightatom->getType()->getAtomicNumber();
      } else { // if not element yet set, set to hydrogen
        elementno = 1;
      }
      break;
    case right:
      if (_bond.leftatom->getType() != NULL) {
        elementno = _bond.leftatom->getType()->getAtomicNumber();
      } else { // if not element yet set, set to hydrogen
        elementno = 1;
      }

      break;
    default:
      ASSERT(0,
          "GLMoleculeObject_bond::GLMoleculeObject_bond() - side is not a valid argument: "+toString(BondSide)+".");
      break;
  }

  QGLMaterial *elementmaterial = getMaterial(elementno);
  setMaterial(elementmaterial);

  resetPosition();
  resetWidth();
}

GLMoleculeObject_bond::~GLMoleculeObject_bond()
{
  // sign off
  switch (BondSide) {
    case left:
      emit BondRemoved(leftatomId, rightatomId);
      break;
    case right:
      emit BondRemoved(rightatomId, leftatomId);
      break;
    default:
      ASSERT(0,
          "GLMoleculeObject_bond::subjectKilled() - side is not a valid argument: "
          +toString(BondSide)+".");
      break;
  }
  _bond.signOff(this, BondObservable::BondRemoved);
  _bond.signOff(this, BondObservable::DegreeChanged);
  leftatom->signOff(this, AtomObservable::PositionChanged);
  leftatom->signOff(this, AtomObservable::ElementChanged);
  rightatom->signOff(this, AtomObservable::PositionChanged);
  rightatom->signOff(this, AtomObservable::ElementChanged);
  LOG(3, "DEBUG: Destroying  GLMoleculeObject_bond to bond " << &_bond << " and side " << BondSide << ".");
}

void GLMoleculeObject_bond::update(Observable *publisher)
{
#ifdef LOG_OBSERVER
  if (publisher == static_cast<const Observable *>(&_bond)) {
    observerLog().addMessage() << "++ Update of Observer "
        << observerLog().getName(static_cast<Observer*>(this))
        << " from bond.";
  } else if (publisher == leftatom) {
    observerLog().addMessage() << "++ Update of Observer "
        << observerLog().getName(static_cast<Observer*>(this))
        << " from leftatom " << _bond.leftatom->getId() << ".";
  } else if (publisher == rightatom) {
    observerLog().addMessage() << "++ Update of Observer " <<
        observerLog().getName(static_cast<Observer*>(this))
        << " from rightatom " << _bond.rightatom->getId() << ".";
  } else
    observerLog().addMessage() << "++ Update of Observer " <<
    observerLog().getName(static_cast<Observer*>(this)) << " from unknown source.";
#endif
}

void GLMoleculeObject_bond::subjectKilled(Observable *publisher)
{
  delete this;
}

void GLMoleculeObject_bond::recieveNotification(Observable *publisher, Notification_ptr notification)
{
#ifdef LOG_OBSERVER
  if (publisher == static_cast<const Observable *>(&_bond)) {
    observerLog().addMessage() << "++ Update of Observer "
        << observerLog().getName(static_cast<Observer*>(this))
        << " received notification from bond for channel "
        << notification->getChannelNo() << ".";
  } else if (publisher == leftatom) {
    observerLog().addMessage() << "++ Update of Observer "
        << observerLog().getName(static_cast<Observer*>(this))
        << " received notification from leftatom " << _bond.leftatom->getId() << " for channel "
        << notification->getChannelNo() << ".";
  } else if (publisher == rightatom) {
    observerLog().addMessage() << "++ Update of Observer "
        << observerLog().getName(static_cast<Observer*>(this))
        << " received notification from rightatom " << _bond.rightatom->getId() << " for channel "
        << notification->getChannelNo() << ".";
  } else
    observerLog().addMessage() << "++ Update of Observer "
        << observerLog().getName(static_cast<Observer*>(this))
        << " received notification from unknown source.";
#endif
  bool DoResetPosition = false;
  bool DoResetWidth = false;
  if (publisher == static_cast<const Observable *>(&_bond)){
    switch (notification->getChannelNo()) {
    case BondObservable::BondRemoved:
      delete this;
      break;
    case BondObservable::DegreeChanged:
      DoResetWidth = true;
      break;
    default:
      ASSERT(0, "GLMoleculeObject_bond::recieveNotification() - unknown signal.");
      break;
    }
  } else {
    // from an atom
    switch (notification->getChannelNo()) {
    case AtomObservable::PositionChanged:
      LOG(2, "INFO: Received notification of PositionChanged.");
      DoResetPosition = true;
      break;
    case AtomObservable::ElementChanged:
      LOG(2, "INFO: Received notification of ElementChanged.");
      DoResetPosition = true;
      break;
    default:
      break;
    }
  }
  if (DoResetPosition) {
    resetPosition();
    emit changed();
  }
  if (DoResetWidth) {
    resetWidth();
    emit changed();
  }
}

void GLMoleculeObject_bond::resetWidth()
{
  const double factor = 1.0f+.5f*(_bond.getDegree()-1);
  LOG(2, "DEBUG: GLMoleculeObject_bond::resetWidth() - setting bond's width to " << factor << ".");
  setScaleX(factor);
  setScaleY(factor);
}

void GLMoleculeObject_bond::resetPosition()
{
  Vector Position;
  Vector OtherPosition;
  switch (BondSide) {
    case left:
      Position = _bond.leftatom->getPosition();
      OtherPosition = _bond.rightatom->getPosition();
      break;
    case right:
      Position = _bond.rightatom->getPosition();
      OtherPosition = _bond.leftatom->getPosition();
      break;
    default:
      ASSERT(0,
          "GLMoleculeObject_bond::resetPosition() - side is not a valid argument: "+toString(BondSide)+".");
      break;
  }
  const double distance =
      Position.distance(OtherPosition)/2.;
  setScaleZ(distance);

  // calculate position
  Vector Z(unitVec[2]); // cylinder are initially aligned along the Z axis
  Vector zeroVec(0.,0.,0.);
  Vector a,b;
  Vector OtherAxis;
  double alpha;
  a = Position - OtherPosition;
  // construct rotation axis
  b = a;
  b.VectorProduct(Z);
  Line axis(zeroVec, b);
  // calculate rotation angle
  alpha = a.Angle(Z);
  // construct other axis to check right-hand rule
  OtherAxis = b;
  OtherAxis.VectorProduct(Z);
  // assure right-hand rule for the rotation
  if (a.ScalarProduct(OtherAxis) < MYEPSILON)
    alpha = M_PI-alpha;
  // check
  Vector a_rotated = axis.rotateVector(a, alpha);
  LOG(3, "INFO: Created cylinder from "// << Position << " to " << OtherPosition
      << a << " to " << a_rotated << " around " << b << " by " << alpha/M_PI*180. << ", respectively.");

  // set position (cylinder offset is in its barymetric center)
  Vector OneFourth(Position - 0.75 * a);
  setPosition(QVector3D(OneFourth[0], OneFourth[1], OneFourth[2]));
  setRotationVector(QVector3D(b[0], b[1], b[2]));
  setRotationAngle(alpha/M_PI*180.);
}
