/*
 * OrthogonalFullSummator.hpp
 *
 *  Created on: 08.09.2012
 *      Author: heber
 */

#ifndef ORTHOGONALFULLSUMMATOR_HPP_
#define ORTHOGONALFULLSUMMATOR_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <boost/fusion/sequence.hpp>

#include "Fragmentation/Summation/OrthogonalSummation.hpp"
#include "Fragmentation/Summation/SetValue.hpp"
#include "Jobs/MPQCDataMap.hpp"

/** OrthogonalFullSummator is a general class for making us of OrthogonalSummation.
 *
 *
 * This class is very similar to OrthogonalSummation only that we specifically include
 * the full set (with a value!) as well. Also there is a function to obtain the
 * contribution. This is intended to check a full calculation against the truncated
 * sum.
 */
template <typename MapType, typename MapKey>
struct OrthogonalFullSummator {
  /** We retrieve the type of the MPQCData member variable from the
   * boost::fusion::map and stored it in this typedef. Note that for
   * internal types (e.g. MapValue::const_iterator we need to place
   * \b typename before).
   */
  typedef typename boost::fusion::result_of::value_at_key<MapType, MapKey>::type MapValue;

  /** Constructor for class OrthogonalFullSummator.
   *
   * \param _subsetmap map with hierarchy of IndexSet's
   * \param _data MPQCData converted to MPQCDataMap_t type
   * \param _jobids job ids to sum data in correct order
   * \param _container container of IndexSet's such that each set has correct order
   *        to job id and hence to _data.
   * \param _MatrixNrLookup lookup from job id to ordering in above vectors
   */
  OrthogonalFullSummator(
      SubsetMap::ptr &_subsetmap,
      const std::vector<MapType> &_data,
      const std::vector<JobId_t> &_jobids,
      const IndexSetContainer::Container_t &_container,
      std::map< JobId_t, size_t > &_MatrixNrLookup) : /* cannot make this const due to operator[] */
    indices(getSubsets(_container)),
    values(createValues(_data, _jobids, _container, _MatrixNrLookup)),
    OS(indices, values, _subsetmap)
  {
    ASSERT( _data.size() == _container.size(),
        "OrthogonalFullSummator() - data and indices don't have same size.");
    ASSERT( _data.size() == _jobids.size(),
        "OrthogonalFullSummator() - data and ids don't have same size.");
    ASSERT( _jobids.size() == _MatrixNrLookup.size(),
        "OrthogonalFullSummator() - ids and MatrixNrLookup don't have same size.");
  }

  /** Summation operator.
   *
   * Initialises instantiated OrthogonalSummation of the respective type via
   * \a OrthogonalFullSummator::data, uses OrthogonalSummation::operator() to sum and returns
   * the result.
   *
   * \param level up to which level to sum up
   * \return result of OrthogonalSummation for given type from MPQCDataMap_t.
   */
  MapValue operator()(const size_t level)
  {
    // evaluate
    const MapValue result = OS(level);
    return result;
  }

  /** Getter for the remaining contribution of the full set.
   *
   * @return contribution of allindices
   */
  MapValue getFullContribution() const {
    typename SetValueMap<MapValue>::const_iterator iter = --OS.getSetValues().end();
    return iter->second->getContribution();
  }

private:
  /** Tiny helper to create the indices from a given IndexSetContainer.
   *
   * @param container container with IndexSet
   * @return all subsets contained in \a container
   */
  typename OrthogonalSummation<MapValue>::InputSets_t getSubsets(
      const IndexSetContainer::Container_t &container)
  {
    typename OrthogonalSummation<MapValue>::InputSets_t indices(
        container.begin(),
        container.end());
    return indices;
  }

  /** Tiny helper to create the values for the summation in the correct order.
   *
   * @param data
   * @param jobids
   * @param container
   * @param MatrixNrLookup
   * @return
   */
  typename OrthogonalSummation<MapValue>::InputValues_t createValues(
      const std::vector<MapType> &data,
      const std::vector<JobId_t> &jobids,
      const IndexSetContainer::Container_t &container,
      std::map< JobId_t, size_t > &MatrixNrLookup)
  {
    typename OrthogonalSummation<MapValue>::InputValues_t values(container.size());
    typename std::vector<MapType>::const_iterator dataiter = data.begin();
    std::vector<size_t>::const_iterator iditer = jobids.begin();
    for (; dataiter != data.end(); ++dataiter, ++iditer) {
      const MapType &Data = *dataiter;
      const MapValue &value = boost::fusion::at_key<MapKey>(Data);
      values[ MatrixNrLookup[*iditer] ] = value;
    }
    return values;
  }
private:
  //!> created indices for OS such that we may hand over refs
  typename OrthogonalSummation<MapValue>::InputSets_t indices;
  //!> created values for OS such that we may hand over refs
  typename OrthogonalSummation<MapValue>::InputValues_t values;
  //!> Summation instance to use in operator()(level)
  OrthogonalSummation<MapValue> OS;
};


#endif /* ORTHOGONALFULLSUMMATOR_HPP_ */
