/*
 * QSeisXMLParser_plottype.cpp
 *
 *  Created on: Apr 5, 2011
 *      Author: bierbach
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// have this after(!) all Qt includes
#include "CodePatterns/MemDebug.hpp"

#include "CodePatterns/Assert.hpp"
#include "CodePatterns/toString.hpp"

#include "QSeisXMLParser_plottype.hpp"


/** Constructor for class plottype.
 *
 * @param name_pt name
 * @param xlabel_pt label for X axis
 * @param ylabel_pt label for Y axis
 * @param style_pt style of plot
 * @param xunits_pt unit vector for x axis
 * @param yunits_pt unit vector for y axis
 * @param unitnames_pt (unit) names for each unit vector coefficient
 * @param unitmap_pt mapping to internal unit names
 */
plottype::plottype(
    std::string name_pt,
    std::string xlabel_pt,
    std::string ylabel_pt,
    std::string style_pt,
    const UnitsVector &xunits_pt,
    const UnitsVector &yunits_pt,
    const std::vector<std::string> &unitnames_pt,
    const std::map<std::string,std::string> &unitmap_pt) :
  name(name_pt),
  style(style_pt),
  unitnames(unitnames_pt),
  unitmap(unitmap_pt)
{
  labels.insert(make_pair(X, xlabel_pt));
  labels.insert(make_pair(Y, ylabel_pt));
  units.insert(make_pair(X, xunits_pt));
  units.insert(make_pair(Y, yunits_pt));
//  std::cout << "Labels: " << labels << std::endl;
//  std::cout << "Units: " << units << std::endl;
}

/** Explicit copy constructor for class plottype.
 *
 * \note is made private to prevent accidental copying of the instance.
 *
 * @param ref instance to copy values from
 */
plottype::plottype(const plottype &ref) :
    name(ref.name),
    style(ref.style),
    units(ref.units),
    labels(ref.labels),
    unitnames(ref.unitnames),
    unitmap(ref.unitmap)
{
  std::cout << "plottype " << &ref << " has been copied." << std::endl;
}

/** Destructor for plottype.
 *
 */
plottype::~plottype()
{
  labels.clear();
  units.clear();
  unitnames.clear();
  unitmap.clear();
}

/** (Re)-set all values for this plottype instance.
 *
 * @param name_pt name
 * @param xlabel_pt label for X axis
 * @param ylabel_pt label for Y axis
 * @param style_pt style of plot
 * @param xunits_pt unit vector for x axis
 * @param yunits_pt unit vector for y axis
 * @param unitnames_pt (unit) names for each unit vector coefficient
 * @param unitmap_pt mapping to internal unit names
 */
void plottype::set_values(
		std::string name_pt,
		std::string xlabel_pt,
		std::string ylabel_pt,
		std::string style_pt,
		const std::vector<int> &xunits_pt,
		const std::vector<int> &yunits_pt,
		const std::vector<std::string> &unitnames_pt,
		const std::map<std::string, std::string> &unitmap_pt)
{
  name = name_pt;
  style = style_pt;
  unitnames = unitnames_pt;
  unitmap = unitmap_pt;
  labels[X] = xlabel_pt;
  labels[Y] = ylabel_pt;
  units[X] = xunits_pt;
  units[Y] = yunits_pt;
}

/** Comparison operator for two plottype's.
 *
 * We just compare the name against the given one.
 *
 * @param type name to compare against.
 * @return true - name equals \a type, false - not
 */
bool plottype::operator==(std::string type)
{
 if(type==name) return true;
 else return false;
}

/** Getter for name.
 *
 * @return \a name
 */
std::string plottype::get_name() const
{
	return name;
}

/** Getter for label.
 *
 * @param _axis label for which axis
 * @return label description for the desired \a _axis
 */
std::string plottype::get_label(const enum axis _axis) const
{
  LabelPerAxis::const_iterator it = labels.find(_axis);
  ASSERT(it != labels.end(),
      "plottype::get_label() - unknown axis type "+toString(_axis)+".");
	return it->second;
}

/** Getter for style string.
 *
 * @return \a style
 */
std::string plottype::get_style() const
{
	return style;
}

/** Returns a string with the unit for the given \a _axis.
 *
 * \note Because it is declared as a const member function we cannot use
 *       operator[] on the maps, as this function is not const (it inserts
 *       if the key is missing in the map). Hence, we have to operator with
 *       find and const_iterator's.
 *
 * @param _axis
 * @return string containing the full unit
 */
std::string plottype::get_units(const enum axis _axis) const
{
	std::string result;
	UnitPerAxis::const_iterator UnitsArray = units.find(_axis);
	ASSERT(UnitsArray != units.end(),
	    "plottype::get_units() - unknown axis type "+toString(_axis)+".");
	bool firstflag = true;
	for (size_t it = 0; it < (UnitsArray->second).size(); ++it) {
	  UnitNamesMap::const_iterator unitname = unitmap.find(unitnames[it]);
	  ASSERT(unitname != unitmap.end(),
	      "plottype::get_units() - unknown unit name "+unitnames[it]+".");
		if(!(UnitsArray->second).at(it))
		  continue;
		result += ((!firstflag) ? std::string(" ") : std::string("")) + (unitname->second);
		if ((UnitsArray->second)[it] != 1)
		  result += std::string("^") + toString((UnitsArray->second)[it]);
		firstflag = false;
	}
	return result;
}
