/*
 * QSeisXMLParser.cpp
 *
 *  Created on: Apr 5, 2011
 *      Author: bierbach
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <QtCore/QDir>
#include <QtCore/QList>

#include "UIElements/Views/Qt4/Plotting/QSeisPageRegistry.hpp"
#include "UIElements/Views/Qt4/Plotting/QSeisPlotPage.hpp"

#include "QSeisXMLParser.hpp"

// property tree uses alignment new
#include <boost/property_tree/ptree.hpp>
#include <boost/property_tree/xml_parser.hpp>

// have this after(!) all Qt includes
#include "CodePatterns/MemDebug.hpp"

#include <boost/foreach.hpp>
#include <string>
#include <set>
#include <exception>
#include <iostream>
#include <sstream>

#include "CodePatterns/Assert.hpp"

#include "CodePatterns/Singleton_impl.hpp"

/** Constructor for QSeisXMLParser.
 *
 */
QSeisXMLParser::QSeisXMLParser()
{
	unitmap["mass"] = "mass";
	unitmap["length"] = "length";
	unitmap["time"] = "time";
	unitmap["current"] = "current";
	unitmap["temperature"] = "temperature";
	unitnames.push_back("mass");
	unitnames.push_back("length");
	unitnames.push_back("time");
	unitnames.push_back("current");
	unitnames.push_back("temperature");
}

/** Destructor for QSeisXMLParser.
 *
 */
QSeisXMLParser::~QSeisXMLParser()
{
  // remove all types
  for(std::vector<plottype*>::iterator it=types.begin();it!=types.end();++it) {
    delete *it;
  }
  types.clear();
  // remove all curves
  for(std::vector<plotcurve*>::iterator it=curves.begin();it!=curves.end();++it) {
		delete *it;
	}
	curves.clear();
}

/** Returns list of desired files.
 *
 * Desired files are all listed in the parsed .xml file, with prefix full
 * file names are constructed and returned.
 *
 * @param prefix prefix of filenames
 * @return list of all desired files
 */
QStringList QSeisXMLParser::getFiles(QString prefix) {
	    QStringList file_names;
	    qDebug("List of desired curves:");
	    for(size_t i=0;i<(curves.size());i++) {
	      QString name = prefix+QString(".")+QString((*curves[i]).get_suffix().c_str());
	      qDebug("\t%s", name.toAscii().constData());
	    	file_names.append(name);
	    }
	    return file_names;
}

/** Loads an .xml file.
 *
 *  Loads the XML file, then parses types and curves contained.
 *
 * @param file file name
 */
void QSeisXMLParser::load(std::istream& file) {
	curves_by_suffix.clear();
	using boost::property_tree::ptree;
	ptree pt;
	qDebug("Parsing XML file ... ");
	read_xml(file,pt);
	qDebug("Parsing types ... ");
	BOOST_FOREACH(ptree::value_type &i,pt.get_child("seismolo.plottypes")) {
		std::string name=i.second.get<std::string>("name");
		std::string style=i.second.get<std::string>("style");
		std::string xlabel=i.second.get<std::string>("xlabel");
		std::string ylabel=i.second.get<std::string>("ylabel");
		std::vector<int> xunits;
		xunits.push_back(i.second.get<int>("xunits.mass"));
		xunits.push_back(i.second.get<int>("xunits.length"));
		xunits.push_back(i.second.get<int>("xunits.time"));
		xunits.push_back(i.second.get<int>("xunits.current"));
		xunits.push_back(i.second.get<int>("xunits.temperature"));
		std::vector<int> yunits;
		yunits.push_back(i.second.get<int>("yunits.mass"));
		yunits.push_back(i.second.get<int>("yunits.length"));
		yunits.push_back(i.second.get<int>("yunits.time"));
		yunits.push_back(i.second.get<int>("yunits.current"));
		yunits.push_back(i.second.get<int>("yunits.temperature"));
		plottype * tmp = new plottype(name,xlabel,ylabel,style,xunits,yunits, unitnames,unitmap);
		types.push_back(tmp);
		qDebug("\ttype: %s, %p", name.c_str(), types.back());
		types_by_name.insert(std::make_pair(types.back()->get_name(),types.back()));
	}
	qDebug("Parsing curves ... ");
	BOOST_FOREACH(ptree::value_type &i,pt.get_child("seismolo.plotcurves")) {
		std::string name=i.second.get<std::string>("name");
		std::string type=i.second.get<std::string>("type");
		std::string suffix=i.second.get<std::string>("suffix");
		std::string color=i.second.get<std::string>("color");
		int column=i.second.get<int>("column");
		// check that type exists
		ASSERT(types_by_name.count(type),
				"QSeisXMLParser::load() - type "+type+" for curve "+name+" not found.");

		std::stringstream tmp;
		tmp<<column;
		if (suffix[0] == '.') // erase leading dot if present
		  suffix.erase(suffix.begin());
		plotcurve* curv=new plotcurve(name,type,suffix,color,column);
		curves.push_back(curv);
    qDebug("\tcurve: %s", curves.back()->get_name().c_str());
		curves_suff.insert(std::make_pair(suffix+tmp.str(),curv));
		curves_by_suffix.insert(std::make_pair(suffix+tmp.str(),type));
		ASSERT(curves_by_suffix.size()!=0,"curves by suffix is empty!");
	}
}

/** Creates plot page for each \a types.
 *
 */
void QSeisXMLParser::createPlotPages() {
	for(size_t i=0;i<types.size();i++) {
		const std::string name=types[i]->get_name();
		QSeisPageRegistry::getInstance().getByName(name);
	}
}

/** Associates each curve in \a curves with its plot page.
 *
 */
void QSeisXMLParser::associateCurves() {
	for(quint32 i=0;i<(curves.size());i++) {
		std::string temp=(*curves[i]).get_type();
		int u=(*curves[i]).get_column();
		std::stringstream ostrs;
		QString watchPath; //=QSeisFileRegistry::getInstance().getWatchPath();
		ostrs << prefix.toStdString() << (*curves[i]).get_suffix() << u;
		QSeisPlotPage *instance = QSeisPageRegistry::getInstance().getByName(temp);
		instance->addCurve(ostrs.str());
	}
}

/** Getter for type for a given curve.
 *
 * @param name name of curve
 * @return type of curve, i.e. the plot page
 */
QString QSeisXMLParser::getTypeOfName(const QString &name) {
	std::string suffix(name.toStdString());
	int found=suffix.find(".");
	ASSERT(found!=-1,"Name of curve " + suffix + " is invalid.");
	suffix=suffix.substr(found+1,suffix.size()-(found+1));

//	Debugausgabe
	std::map<std::string,std::string>::iterator m;
	for(m=curves_by_suffix.begin();m!=curves_by_suffix.end();++m) {
		std::cerr << m->first << "----" << m->second << std::endl;
	}
	if (curves_by_suffix.count(suffix))
		return QString(curves_by_suffix[suffix].c_str());
	else
		return QString();
}

std::string QSeisXMLParser::getCurveColor(std::string suff_col) {
	if(curves_suff.count(suff_col)!=0) {
		return curves_suff[suff_col]->get_color();
	}
	else return std::string();
}

plottype* QSeisXMLParser::return_plottype_by_name(std::string name) {
	if(types_by_name.count(name)!=0) {
		return types_by_name[name];
	} else return NULL;
}

void QSeisXMLParser::set_units(const std::vector<std::string>& unitnames_xml,const std::map<std::string,std::string>& unitmap_xml) {
	unitmap=unitmap_xml;unitnames=unitnames_xml;
}

CONSTRUCT_SINGLETON(QSeisXMLParser)
