/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * KeySetsContainerUnitTest.cpp
 *
 *  Created on: Sep 16, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

// include headers that implement a archive in simple text format
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>

#include <boost/assign.hpp>

#include "Fragmentation/KeySetsContainer.hpp"

#include "KeySetsContainerUnitTest.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

using namespace boost::assign;

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( KeySetsContainerTest );


void KeySetsContainerTest::setUp()
{
  Keys = new KeySetsContainer();
};


void KeySetsContainerTest::tearDown()
{
  delete Keys;
};

/** UnitTest for operator==()
 */
void KeySetsContainerTest::EqualityTest()
{
  // compare to self
  CPPUNIT_ASSERT( *Keys == *Keys );

  // create other instance
  KeySetsContainer *Keys2 = new KeySetsContainer();
  CPPUNIT_ASSERT( *Keys == *Keys2 );
  Keys2->Order = 2;
  CPPUNIT_ASSERT( *Keys != *Keys2 );
  delete Keys2;
};

/** UnitTest for serialization
 */
void KeySetsContainerTest::SerializationTest()
{
  // write element to stream
  std::stringstream stream;
  boost::archive::text_oarchive oa(stream);
  oa << Keys;

  std::cout << "Contents of archive is " << stream.str() << std::endl;

  // create and open an archive for input
  boost::archive::text_iarchive ia(stream);
  // read class state from archive
  KeySetsContainer *Keys2;

  ia >> Keys2;

  CPPUNIT_ASSERT (*Keys == *Keys2);

  delete Keys2;
};

static void fillKeySetsContainer(KeySetsContainer *_Keys)
{
  // insert keysets
  KeySetsContainer::IntVector keyset;
  keyset += 1;
  _Keys->KeySets.push_back(keyset);
  keyset += 2;
  _Keys->KeySets.push_back(keyset);
  keyset += 3;
  _Keys->KeySets.push_back(keyset);
  // insert number of keys per keyset
  _Keys->AtomCounter += 1,2,3;
  // insert total number of keysets/fragments
  _Keys->FragmentCounter = 3;
  // insert maximum number of keys over all contained keysets
  _Keys->Order = 3;
  // insert number of keysets with the same number of keys
  _Keys->FragmentsPerOrder += 1,1,1;
  // insert keyset index in keysets per same number of keys
  _Keys->OrderSet.resize(3);
  _Keys->OrderSet[0] = KeySetsContainer::IntVector(1, 0);
  _Keys->OrderSet[1] = KeySetsContainer::IntVector(1, 1);
  _Keys->OrderSet[2] = KeySetsContainer::IntVector(1, 2);
}

/** UnitTest for insert() into an empty container
 */
void KeySetsContainerTest::insertIntoEmptyTest()
{
  // fill other container
  KeySetsContainer *Keys2 = new KeySetsContainer();
  fillKeySetsContainer(Keys2);

  // insert filled one (Keys2) into empty (Keys)
  Keys->insert(*Keys2);

  CPPUNIT_ASSERT (*Keys == *Keys2);

  delete Keys2;
}

/** UnitTest for insert() empty into a filled container
 */
void KeySetsContainerTest::insertEmptyIntoTest()
{
  // fill other container
  KeySetsContainer *Keys2 = new KeySetsContainer();
  fillKeySetsContainer(Keys2);

  // create result container (we know this works from insertIntoEmptyTest())
  KeySetsContainer *resultKeys = new KeySetsContainer();
  resultKeys->insert(*Keys2);

  // fill empty (Keys) into filled oned (Keys2)
  Keys2->insert(*Keys);

  // create empty for comparison
  KeySetsContainer *emptyKeys = new KeySetsContainer();

  CPPUNIT_ASSERT (*Keys2 == *resultKeys);
  CPPUNIT_ASSERT (*Keys == *emptyKeys);

  delete Keys2;
  delete resultKeys;
  delete emptyKeys;
}

/** UnitTest for insert() smaller filled into another filled container
 */
void KeySetsContainerTest::insertSmallerIntoTest()
{
  // fill other container
  KeySetsContainer *Keys2 = new KeySetsContainer();
  fillKeySetsContainer(Keys2);

  // fill Keys, too
  {
    KeySetsContainer::IntVector keyset;
    keyset += 2;
    Keys->KeySets.push_back(keyset);
    // insert number of keys per keyset
    Keys->AtomCounter += 1;
    // insert total number of keysets/fragments
    Keys->FragmentCounter = 1;
    // insert maximum number of keys over all contained keysets
    Keys->Order = 1;
    // insert number of keysets with the same number of keys
    Keys->FragmentsPerOrder += 1;
    // insert keyset index in keysets per same number of keys
    Keys->OrderSet.resize(1);
    Keys->OrderSet[0] = KeySetsContainer::IntVector(1, 0);
  }

  // create comparison container
  KeySetsContainer *resultKeys = new KeySetsContainer();
  {
    fillKeySetsContainer(resultKeys);
    KeySetsContainer::IntVector keyset;
    keyset += 2;
    resultKeys->KeySets.push_back(keyset);
    resultKeys->AtomCounter += 1;
    resultKeys->FragmentCounter = 4;
    resultKeys->Order = 3;
    resultKeys->FragmentsPerOrder[0] = 2;
    CPPUNIT_ASSERT( resultKeys->OrderSet.size() == (size_t)3 );
    resultKeys->OrderSet[0] += 3;
  }

  // insert smaller into larger one
  Keys2->insert(*Keys);

  CPPUNIT_ASSERT (*Keys != *Keys2);
  CPPUNIT_ASSERT (*resultKeys == *Keys2);

  delete Keys2;
  delete resultKeys;
}

/** UnitTest for insert() larger filled into another filled container
 */
void KeySetsContainerTest::insertLargerIntoTest()
{
  // fill other container
  KeySetsContainer *Keys2 = new KeySetsContainer();
  fillKeySetsContainer(Keys2);

  // fill Keys, too
  {
    KeySetsContainer::IntVector keyset;
    keyset += 2;
    Keys->KeySets.push_back(keyset);
    // insert number of keys per keyset
    Keys->AtomCounter += 1;
    // insert total number of keysets/fragments
    Keys->FragmentCounter = 1;
    // insert maximum number of keys over all contained keysets
    Keys->Order = 1;
    // insert number of keysets with the same number of keys
    Keys->FragmentsPerOrder += 1;
    // insert keyset index in keysets per same number of keys
    Keys->OrderSet.resize(1);
    Keys->OrderSet[0] = KeySetsContainer::IntVector(1, 0);
  }

  // create comparison container
  KeySetsContainer *resultKeys = new KeySetsContainer();
  {
    // we know this works from insertIntoEmptyTest()
    resultKeys->insert(*Keys);
    // add larger keysets
    fillKeySetsContainer(resultKeys);

    resultKeys->AtomCounter.clear();
    resultKeys->AtomCounter += 1,1,2,3;
    resultKeys->FragmentCounter = 4;
    resultKeys->FragmentsPerOrder.clear();
    resultKeys->FragmentsPerOrder += 2,1,1;
    resultKeys->OrderSet.clear();
    resultKeys->OrderSet.resize(3);
    resultKeys->OrderSet[0] = KeySetsContainer::IntVector(1, 0);
    resultKeys->OrderSet[0] += 1;
    resultKeys->OrderSet[1] = KeySetsContainer::IntVector(1, 2);
    resultKeys->OrderSet[2] = KeySetsContainer::IntVector(1, 3);
  }

  // insert larger into smaller one
  Keys->insert(*Keys2);

  CPPUNIT_ASSERT (*Keys != *Keys2);
  CPPUNIT_ASSERT (*resultKeys == *Keys);

  delete Keys2;
  delete resultKeys;
}
