/*
 * MPQCData.hpp
 *
 *  Created on: Feb 08, 2012
 *      Author: heber
 */

#ifndef MPQCDATA_HPP_
#define MPQCDATA_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <boost/serialization/access.hpp>
#include <boost/serialization/vector.hpp>
#include <boost/serialization/version.hpp>

#include <iosfwd>
#include <vector>

#include "Fragmentation/Summation/SetValues/SamplingGrid.hpp"
#include "Fragmentation/Summation/SetValues/FragmentForces.hpp"

class MPQCCommandJob;
class MPQCCommandJobTest;
class MPQCDataTest;

/** Internal class that holds the data and can be serialized.
 *
 */
class MPQCData {
  //!> allow MPQCCommandJob access to member variables directly
  friend class MPQCCommandJob;
  //!> grant MPQCCommandJob's unit test access
  friend class MPQCCommandJobTest;
  //!> grant unit test access
  friend class MPQCDataTest;
  //!> grant access to output stream operator
  friend std::ostream & operator<<(std::ostream &ost, const MPQCData &data);
public:
  /** Constructor for class MPQCData with full sampling information.
   *
   * \param _props properties of the grid
   */
  MPQCData(const SamplingGridProperties &_props);

  /** Default Constructor for class MPQCData.
   *
   */
  MPQCData();

  bool operator==(const MPQCData &other) const;

  bool operator!=(const MPQCData &other) const {
    return !(*this == other);
  }

  /** Assignment operator with a downsampled grid.
   *
   * All values are taken over (with self-assignment check), but the grid
   * is sampled down to the desired \a _level (left untouched if larger
   * than grid level contained in \a other).
   *
   * \param instance instance to assign, containing desired grid and level
   * \param other instance to get values and grid from
   */
  static void assignWithDownsampledGrid(
      MPQCData &instance,
      const MPQCData &other);

  /// Energie structure
  struct energy_t {
    /** Constructor for struct energy_t, sets all to zero.
     *
     */
    energy_t();

    double total;
    double nuclear_repulsion;
    double electron_coulomb;
    double electron_exchange;
    double correlation;
    double overlap;
    double kinetic;
    double hcore;

    std::vector<double> eigenvalues;
  } energies;

  /// Forces
  FragmentForces forces;

  //!> whether to actually sample the density
  enum DoLongrange_t {
    DontSampleDensity=0,
    DoSampleDensity=1
  };
  DoLongrange_t DoLongrange;

  //!> whether to sample just the valence or the total electron and nuclei density
  enum DoValenceOnly_t {
    DontSampleValenceOnly=0,
    DoSampleValenceOnly=1
  };
  DoValenceOnly_t DoValenceOnly;

  /// Density
  SamplingGrid sampled_grid;

  // nuclei positions and charges
  std::vector< std::vector<double> > positions;
  std::vector<double> charges;

  /// Timing structure
  struct times_t {
    /** Constructor for struct times_t, sets all to zero.
     *
     */
    times_t();

    double total_walltime;
    double total_cputime;
    double total_flops;
    double gather_walltime;
    double gather_cputime;
    double gather_flops;
  } times;

  //!> reached (relative) accuracy
  double accuracy;
  //!> desired accuracy
  double desired_accuracy;

private:
  friend class boost::serialization::access;
  // serialization
  template <typename Archive>
  void serialize(Archive& ar, const unsigned int version)
  {
    ar & energies.total;
    ar & energies.nuclear_repulsion;
    ar & energies.electron_coulomb;
    ar & energies.electron_exchange;
    ar & energies.correlation;
    ar & energies.overlap;
    ar & energies.kinetic;
    ar & energies.hcore;
    ar & energies.eigenvalues;
    if (version < 2) {
      FragmentForces::forces_t justforces(forces);
      ar & justforces;
      dynamic_cast<FragmentForces::forces_t &>(forces) = justforces;
    } else
      ar & forces;
    ar & sampled_grid;
    ar & DoLongrange;
    if (version > 0)
      ar & DoValenceOnly;
    ar & positions;
    ar & charges;
    ar & times.total_walltime;
    ar & times.total_cputime;
    ar & times.total_flops;
    ar & times.gather_walltime;
    ar & times.gather_cputime;
    ar & times.gather_flops;
    if (version > 2) {
      ar & accuracy;
      ar & desired_accuracy;
    }
  }
};

BOOST_CLASS_VERSION(MPQCData, 3)

std::ostream & operator<<(std::ostream &ost, const MPQCData &data);

#endif /* MPQCDATA_HPP_ */
