/*
 * Woodcock.cpp
 *
 *  Created on: Aug 20, 2010
 *      Author: crueger
 */

#include "Woodcock.hpp"

#include "element.hpp"
#include "config.hpp"
#include "Helpers/Verbose.hpp"
#include "Helpers/Log.hpp"
#include "ThermoStatContainer.hpp"

#include <sstream>

Woodcock::Woodcock(int _ScaleTempStep) :
  ScaleTempStep(_ScaleTempStep)
{}

Woodcock::Woodcock() :
  ScaleTempStep(25)
{}

Woodcock::~Woodcock()
{}

const char *ThermostatTraits<Woodcock>::name = "Woodcock";

std::string ThermostatTraits<Woodcock>::getName(){
  return ThermostatTraits<Woodcock>::name;
}

Thermostat *ThermostatTraits<Woodcock>::make(class ConfigFileBuffer * const fb){
  int ScaleTempStep;
  const int verbose = 0;
  ParseForParameter(verbose,fb,"Thermostat", 0, 2, 1, int_type, &ScaleTempStep, 1, critical); // read scaling frequency
  return new Woodcock(ScaleTempStep);
}

double Woodcock::scaleAtoms(unsigned int step,double ActualTemp,ATOMSET(std::list) atoms){
  return doScaleAtoms(step,ActualTemp,atoms.begin(),atoms.end());
}

double Woodcock::scaleAtoms(unsigned int step,double ActualTemp,ATOMSET(std::vector) atoms){
  return doScaleAtoms(step,ActualTemp,atoms.begin(),atoms.end());
}

double Woodcock::scaleAtoms(unsigned int step,double ActualTemp,ATOMSET(std::set) atoms){
  return doScaleAtoms(step,ActualTemp,atoms.begin(),atoms.end());
}

template <class ForwardIterator>
double Woodcock::doScaleAtoms(unsigned int step,double ActualTemp,ForwardIterator begin,ForwardIterator end){
  double ekin=0;
  if ((ScaleTempStep > 0) && ((step-1) % ScaleTempStep == 0)) {
    double ScaleTempFactor = sqrt(getContainer().TargetTemp/ActualTemp);
    DoLog(2) && (Log() << Verbose(2) <<  "Applying Woodcock thermostat..." << endl);
    double ekin;
    for (ForwardIterator iter = begin; iter!=end;++iter){
      Vector &U = (*iter)->Trajectory.U.at(step);
      if ((*iter)->FixedIon == 0){ // even FixedIon moves, only not by other's forces
        U *= ScaleTempFactor;
        ekin += 0.5*(*iter)->getType()->getMass() * U.NormSquared();
      }
    }
  }
  return ekin;
}

std::string Woodcock::name(){
  return ThermostatTraits<Woodcock>::name;
}

std::string Woodcock::writeParams(){
  std::stringstream sstr;
  sstr << ScaleTempStep;
  return sstr.str();
}
