/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * MoleculeUnitTest.cpp
 *
 *  Created on: Mar 19, 2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include <algorithm>

#include "Atom/atom.hpp"
#include "CodePatterns/Assert.hpp"
#include "Element/element.hpp"
#include "Element/periodentafel.hpp"
#include "molecule.hpp"

#include "MoleculeUnitTest.hpp"


#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( MoleculeUnittest );

size_t MoleculeUnittest::MaxAtoms = 6;

void MoleculeUnittest::setUp(){
  // failing asserts should be thrown
  ASSERT_DO(Assert::Throw);

  atomVector.resize((size_t)MaxAtoms);
  std::generate_n(atomVector.begin(), MaxAtoms,
      boost::bind(&World::createAtom, boost::ref(World::getInstance())));
  std::for_each(atomVector.begin(), atomVector.end(),
      boost::bind(static_cast<void (AtomInfo::*)(int)>(&AtomInfo::setType), _1, (atomicNumber_t)1));

  mol = new molecule;
  std::for_each(atomVector.begin(), atomVector.end(),
      boost::bind(&molecule::AddAtom, boost::ref(mol), _1));
}

void MoleculeUnittest::tearDown()
{
  delete mol;
  World::purgeInstance();
}

/** Unit test for molecule::getBoundingShape() with a linear chain
 *
 */
void MoleculeUnittest::getBoundingShapeTest_linearchain()
{
  // prepare a chain of atoms
  double offset = 0.;
  BOOST_FOREACH(atom *_atom, atomVector) {
    _atom->setPosition( Vector(offset, 0., 0.) );
    offset += 1.;
  }

  {
    // get bounding shape
    Shape s = mol->getBoundingShape();

    // check that each atom is truely inside the shape
    BOOST_FOREACH(atom *_atom, atomVector) {
      CPPUNIT_ASSERT( s.isInside(_atom->getPosition()) );
    }
  }
}

/** Unit test for molecule::getBoundingShape() with a v-shaped molecule.
 *
 */
void MoleculeUnittest::getBoundingShapeTest_vshaped()
{
  double xoffset = -2.5;
  double yoffset = -2.5;
  double yadder = -1;
  BOOST_FOREACH(atom *_atom, atomVector) {
    _atom->setPosition( Vector(xoffset, yoffset, 0.) );
    xoffset += 1.;
    yoffset -= yadder;
    if (yoffset <= 0) {
      yadder = 1.;
    }
  }

  {
    // get bounding shape
    Shape s = mol->getBoundingShape();

    // check that each atom is truely inside the shape
    BOOST_FOREACH(atom *_atom, atomVector) {
      CPPUNIT_ASSERT( s.isInside(_atom->getPosition()) );
    }
  }
}
