/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * AtomIdSetUnitTest.cpp
 *
 *  Created on: Feb 21, 2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include <algorithm>

#include "Atom/atom.hpp"
#include "AtomIdSet.hpp"
#include "CodePatterns/Assert.hpp"
#include "World.hpp"

#include "AtomIdSetUnitTest.hpp"


#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( AtomIdSetUnittest );

size_t AtomIdSetUnittest::MaxAtoms = 6;

void AtomIdSetUnittest::setUp(){
  // failing asserts should be thrown
  ASSERT_DO(Assert::Throw);

  atomVector.resize((size_t)MaxAtoms);
  std::generate_n(atomVector.begin(), MaxAtoms,
      boost::bind(&World::createAtom, boost::ref(World::getInstance())));
}

void AtomIdSetUnittest::tearDown()
{
  World::purgeInstance();
}

void AtomIdSetUnittest::inserteraseTest()
{
  // insert all
  for (size_t i=0; i< MaxAtoms; ++i) {
    atoms.insert(atomVector[i]);
  }
  CPPUNIT_ASSERT_EQUAL( (size_t)MaxAtoms, atoms.size() );

  // erase them again
  for (size_t i=0; i< MaxAtoms; ++i) {
    atoms.erase(atomVector[i]);
  }
  CPPUNIT_ASSERT_EQUAL( (size_t)0, atoms.size() );

  {
    // use atomVector in cstor
    AtomIdSet otherAtoms(atomVector);
    CPPUNIT_ASSERT_EQUAL( (size_t)MaxAtoms, otherAtoms.size() );
  }
}

void AtomIdSetUnittest::positionTest()
{
  // non-const iterators
  CPPUNIT_ASSERT( atoms.begin() == atoms.end() );

  // const iterators
  {
    AtomIdSet::const_iterator beg_iter = atoms.begin();
    AtomIdSet::const_iterator end_iter = atoms.end();
    CPPUNIT_ASSERT( beg_iter == end_iter );
  }

  // insert an element
  atoms.insert(atomVector[0]);
  CPPUNIT_ASSERT( atoms.begin() != atoms.end() );

  {
    AtomIdSet::const_iterator beg_iter = atoms.begin();
    AtomIdSet::const_iterator end_iter = atoms.end();
    CPPUNIT_ASSERT( beg_iter != end_iter );
    ++beg_iter;
    CPPUNIT_ASSERT( beg_iter == end_iter );
  }
}

void AtomIdSetUnittest::findTest()
{
  // search for atom
  atoms.insert(atomVector[0]);
  CPPUNIT_ASSERT( atoms.find(atomVector[0]) == atoms.begin() );
  CPPUNIT_ASSERT( atoms.find(atomVector[0]) != atoms.end() );
  atoms.erase(atomVector[0]);

  // search for key
  atoms.insert(atomVector[0]->getId());
  CPPUNIT_ASSERT( atoms.find(atomVector[0]->getId()) == atoms.begin() );
  CPPUNIT_ASSERT( atoms.find(atomVector[0]->getId()) != atoms.end() );
  atoms.erase(atomVector[0]->getId());
}


void AtomIdSetUnittest::emptyTest()
{
  // initially empty
  CPPUNIT_ASSERT( atoms.empty() );

  // filled with one then no more
  atoms.insert(atomVector[0]);
  CPPUNIT_ASSERT( !atoms.empty() );

  // erase and again empty
  atoms.erase(atomVector[0]);
  CPPUNIT_ASSERT( atoms.empty() );
}

void AtomIdSetUnittest::sizeTest()
{
  // initially empty
  CPPUNIT_ASSERT_EQUAL( (size_t)0, atoms.size() );

  // filled with one, then no more
  atoms.insert(atomVector[0]);
  CPPUNIT_ASSERT_EQUAL( (size_t)1, atoms.size() );

  // erase and again empty
  atoms.erase(atomVector[0]);
  CPPUNIT_ASSERT_EQUAL( (size_t)0, atoms.size() );
}

