/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * ApproximateShapeArea.cpp
 *
 *  Created on: Jan 30, 2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include <cmath>
#include <set>
#include <vector>

#include "Atom/TesselPoint.hpp"
#include "LinkedCell/PointCloudAdaptor.hpp"
#include "ApproximateShapeArea.hpp"
#include "Tesselation/tesselation.hpp"

/** Constructor of class ApproximateShapeArea.
 *
 * @param _shape shape to calculate the surface area approximately
 */
ApproximateShapeArea::ApproximateShapeArea(const Shape &_shape) :
		shape(_shape)
{}

/** Destructor of class ApproximateShapeArea.
 *
 */
ApproximateShapeArea::~ApproximateShapeArea()
{}

/** Calculate the approximate surface area of the given \a shape.
 *
 * @return surface area approximated
 */
double ApproximateShapeArea::operator()() const
{
  const double distance = 2.;
  const double min_area = distance*distance*distance;
  const double radius = shape.getRadius();
  const double surfacearea_upperbound = 4.*M_PI * radius*radius;
  const size_t N = (int)(surfacearea_upperbound/min_area);
  // get surface points
  std::vector<Vector> SurfacePoints =
      shape.getHomogeneousPointsOnSurface(N);
  typedef std::set<TesselPoint *> NodeSet;
  NodeSet SurfaceNodes;
  for (size_t i=0;i<SurfacePoints.size();++i) {
    TesselPoint * Walker = new TesselPoint();
    Walker->setName(std::string("SurfaceNode")+toString(i));
    Walker->setPosition(SurfacePoints[i]);
    SurfaceNodes.insert(Walker);
  }

  // tesselate
  PointCloudAdaptor<NodeSet> cloud(&SurfaceNodes, "TesselPointSTLList");
  Tesselation *tesselator = new Tesselation;
  (*tesselator)(cloud, distance);

  // obtain surface area (assume angstroem)
  const double surfacearea = tesselator->getAreaOfEnvelope(true);

  // clear up
  delete tesselator;
  for (NodeSet::iterator iter = SurfaceNodes.begin();
      !SurfaceNodes.empty(); iter = SurfaceNodes.begin()) {
    delete *iter;
    SurfaceNodes.erase(iter);
  }

  return surfacearea;
}

