/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * ShapeOpsUnitTest.cpp
 *
 *  Created on: Jun 18, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include <cmath>
#include <limits>

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

#include "CodePatterns/Assert.hpp"

#include "Helpers/defs.hpp"
#include "LinearAlgebra/RealSpaceMatrix.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "Shapes/BaseShapes.hpp"
#include "Shapes/Shape.hpp"
#include "Shapes/ShapeOps.hpp"

#include "ShapeOpsUnitTest.hpp"

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( ShapeOpsTest );

void ShapeOpsTest::setUp()
{
  // failing asserts should be thrown
  ASSERT_DO(Assert::Throw);
}

void ShapeOpsTest::tearDown()
{}

void ShapeOpsTest::resizeTest()
{
  Shape s = resize(Sphere(), 2.);

  CPPUNIT_ASSERT( s.isInside( Vector(1.5,0.,0.) ) );
  CPPUNIT_ASSERT( s.isInside( Vector(0.,1.5,0.) ) );
  CPPUNIT_ASSERT( s.isInside( Vector(0.,0.,1.5) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(2.,0.,0.) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(0.,2.,0.) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(0.,0.,2.) ) );
}

void ShapeOpsTest::translateTest()
{
  Shape s = translate(Sphere(), Vector(1.,0.,0.));

  CPPUNIT_ASSERT( s.isInside( Vector(1.5,0.,0.) ) );
  CPPUNIT_ASSERT( s.isInside( Vector(1.,.5,0.) ) );
  CPPUNIT_ASSERT( s.isInside( Vector(1.,0.,.5) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(2.,0.,0.) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(1.,1.,0.) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(1.,0.,1.) ) );
  CPPUNIT_ASSERT( !s.isInside( Vector(-.5,0.,0.) ) );
  CPPUNIT_ASSERT( !s.isInside( Vector(0.,-.5,0.) ) );
  CPPUNIT_ASSERT( !s.isInside( Vector(0.,0.,-.5) ) );
}

void ShapeOpsTest::stretchTest()
{
#ifndef NDEBUG
  CPPUNIT_ASSERT_THROW( Shape test = stretch(Sphere(), Vector(-2.,0.,0.)), Assert::AssertionFailure );
#endif
  Shape s = stretch(Sphere(), Vector(2.,1.,1.));

  CPPUNIT_ASSERT( s.isInside( Vector(1.5,0.,0.) ) );
  CPPUNIT_ASSERT( !s.isInside( Vector(0.,1.5,0.) ) );
  CPPUNIT_ASSERT( !s.isInside( Vector(0.,0.,1.5) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(2.,0.,0.) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(0.,1.,0.) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(0.,1.,0.) ) );
}

void ShapeOpsTest::transformTest()
{
  RealSpaceMatrix M;
  M.setRotation(45., 0., 0.);
  Shape s = transform(Sphere(), M);

  CPPUNIT_ASSERT( s.isInside( Vector(.5,0.,0.) ) );
  CPPUNIT_ASSERT( s.isInside( Vector(0.,.5,0.) ) );
  CPPUNIT_ASSERT( s.isInside( Vector(0.,0.,.5) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(1.,0.,0.) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(0.,1.,0.) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(0.,0.,1.) ) );
}

void ShapeOpsTest::getCenterTest()
{
  Shape s = Sphere();
  Shape t = Sphere(Vector(2.,0.,0.),1.);
  Vector offset(1.,0.,0);
  RealSpaceMatrix M;
  M.setRotation(45.,0.,0.);

  // Sphere
  {
    // sphere at origin
    CPPUNIT_ASSERT_EQUAL( Vector(0.,0.,0), s.getCenter() );

    // translated sphere
    CPPUNIT_ASSERT_EQUAL( offset, translate(s, offset).getCenter() );

    // resized sphere
    CPPUNIT_ASSERT_EQUAL( Vector(0.,0.,0), resize(s, 2.).getCenter() );

    // stretched sphere
    CPPUNIT_ASSERT_EQUAL( Vector(0.,0.,0), stretch(s, Vector(2.,1.,1.)).getCenter() );

    // transformed sphere
    CPPUNIT_ASSERT_EQUAL( Vector(0.,0.,0), transform(s, M).getCenter() );

    // resized and translated sphere
    CPPUNIT_ASSERT_EQUAL( offset, translate(resize(s, 2.), offset).getCenter() );
  }

  // AND spheres
  {
    // sphere at origin
    CPPUNIT_ASSERT_EQUAL( Vector(1.,0.,0), (s && t).getCenter() );

    // translated sphere
    CPPUNIT_ASSERT_EQUAL( Vector(2.,0.,0), translate((s && t), offset).getCenter() );

    // resized sphere
    CPPUNIT_ASSERT_EQUAL( Vector(1.,0.,0), resize((s && t), 2.).getCenter() );

    // stretched sphere
    CPPUNIT_ASSERT_EQUAL( Vector(1.5,0.,0), (stretch(s, Vector(2.,1.,1.)) && t).getCenter() );

    // transformed sphere
    CPPUNIT_ASSERT_EQUAL( Vector(1.,0.,0), (transform(s, M) && t).getCenter() );

    // resized and translated sphere
    CPPUNIT_ASSERT_EQUAL( Vector(2.,0.,0), translate(resize((s && t), 2.), offset).getCenter() );
  }

  // OR spheres
  {
    // sphere at origin
    CPPUNIT_ASSERT_EQUAL( Vector(1.,0.,0), (s || t).getCenter() );

    // translated sphere
    CPPUNIT_ASSERT_EQUAL( Vector(2.,0.,0), translate((s || t), offset).getCenter() );

    // resized sphere
    CPPUNIT_ASSERT_EQUAL( Vector(1.,0.,0), resize((s || t), 2.).getCenter() );

    // stretched sphere
    CPPUNIT_ASSERT_EQUAL( Vector(0.5,0.,0), (stretch(s, Vector(2.,1.,1.)) || t).getCenter() );

    // transformed sphere
    CPPUNIT_ASSERT_EQUAL( Vector(1.,0.,0), (transform(s, M) || t).getCenter() );

    // resized and translated sphere
    CPPUNIT_ASSERT_EQUAL( Vector(2.,0.,0), translate(resize((s || t), 2.), offset).getCenter() );
  }

  // NOT spheres
  {
    // sphere at origin
    CPPUNIT_ASSERT_EQUAL( Vector(0.,0.,0), (!s).getCenter() );
  }
}

void ShapeOpsTest::getRadiusTest()
{
  Shape s = Sphere();
  Shape t = Sphere(Vector(2.,0.,0.),1.);
  Vector offset(1.,0.,0);
  RealSpaceMatrix M;
  M.setRotation(45.,0.,0.);

  // Sphere
  {
    // sphere at origin
    CPPUNIT_ASSERT_EQUAL( 1., s.getRadius() );

    // translated sphere
    CPPUNIT_ASSERT_EQUAL( 1., translate(s, offset).getRadius() );

    // resized sphere
    CPPUNIT_ASSERT_EQUAL( 2., resize(s, 2.).getRadius() );

    // stretched sphere
    CPPUNIT_ASSERT_EQUAL( 2., stretch(s, Vector(2.,1.,1.)).getRadius() );

    // transformed sphere
    CPPUNIT_ASSERT_EQUAL( 1., transform(s, M).getRadius() );

    // resized and translated sphere
    CPPUNIT_ASSERT_EQUAL( 2., translate(resize(s, 2.), offset).getRadius() );
  }

  // AND spheres
  {
    // sphere at origin
    CPPUNIT_ASSERT_EQUAL( 1., (s && t).getRadius() );

    // translated sphere
    CPPUNIT_ASSERT_EQUAL( 1., translate((s && t), offset).getRadius() );

    // resized sphere
    CPPUNIT_ASSERT_EQUAL( 2., resize((s && t), 2.).getRadius() );

    // stretched sphere
    CPPUNIT_ASSERT_EQUAL( 1., (stretch(s, Vector(2.,1.,1.)) && t).getRadius() );

    // transformed sphere
    CPPUNIT_ASSERT_EQUAL( 1., (transform(s, M) && t).getRadius() );

    // resized and translated sphere
    CPPUNIT_ASSERT_EQUAL( 2., translate(resize((s && t), 2.), offset).getRadius() );
  }

  // OR spheres
  {
    // sphere at origin
    CPPUNIT_ASSERT_EQUAL( 2., (s || t).getRadius() );

    // translated sphere
    CPPUNIT_ASSERT_EQUAL( 2., translate((s || t), offset).getRadius() );

    // resized sphere
    CPPUNIT_ASSERT_EQUAL( 4., resize((s || t), 2.).getRadius() );

    // stretched sphere
    CPPUNIT_ASSERT_EQUAL( 2.5, (stretch(s, Vector(2.,1.,1.)) || t).getRadius() );

    // transformed sphere
    CPPUNIT_ASSERT_EQUAL( 2., (transform(s, M) || t).getRadius() );

    // resized and translated sphere
    CPPUNIT_ASSERT_EQUAL( 4., translate(resize((s || t), 2.), offset).getRadius() );
  }

  // NOT spheres
  {
    // sphere at origin
    CPPUNIT_ASSERT_EQUAL( std::numeric_limits<double>::infinity(), (!s).getRadius() );
  }
}

