/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * Ops_FillPredicateUnitTest.cpp
 *
 *  Created on: Jan 19, 2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "CodePatterns/Assert.hpp"

#include "Atom/atom.hpp"
#include "Box.hpp"
#include "Filling/Predicates/IsValidInDomain_FillPredicate.hpp"
#include "Filling/Predicates/IsVoidNode_FillPredicate.hpp"
#include "Filling/Predicates/Ops_FillPredicate.hpp"
#include "LinearAlgebra/RealSpaceMatrix.hpp"
#include "Filling/NodeTypes.hpp"
#include "Shapes/BaseShapes.hpp"
#include "World.hpp"

#include "Ops_FillPredicateUnitTest.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( Ops_FillPredicateTest );

void DoSomething(atom * const _atom)
{
  _atom->setPosition( Vector((double)_atom->getId(), 0., 0.) );
}

void Ops_FillPredicateTest::setUp()
{
  // failing asserts should be thrown
  ASSERT_DO(Assert::Throw);

  // set default BCs to ignore
  World::getInstance().getDomain().setConditions(
      BoundaryConditions::Conditions_t(3, BoundaryConditions::Ignore)
  );

  Domainpredicate = new FillPredicate(IsValidInDomain_FillPredicate(World::getInstance().getDomain()));

  // create some atoms as "neighbours"
  atoms.resize((size_t)5, NULL);
  std::generate_n(atoms.begin(), (size_t)5, boost::bind(&World::createAtom, World::getPointer()) );

  // position them
  std::for_each(atoms.begin(), atoms.end(), &DoSomething);

  Voidpredicate = new FillPredicate(IsVoidNode_FillPredicate(Sphere()));
}


void Ops_FillPredicateTest::tearDown()
{
  delete Domainpredicate;
  delete Voidpredicate;

  World::purgeInstance();
}

/** Test whether operator() returns as desired
 *
 */
void Ops_FillPredicateTest::AndTest()
{
	FillPredicate Andpredicate = (*Domainpredicate) && (*Voidpredicate);

	CPPUNIT_ASSERT( !Andpredicate( Vector(-1.,1.,0.)) );
  CPPUNIT_ASSERT( !Andpredicate( Vector(-2.,0.,0.)) );
	CPPUNIT_ASSERT( Andpredicate( Vector(5.,1.,0.)) );
  CPPUNIT_ASSERT( Andpredicate( Vector(5.,0.,1.)) );
  for (double i = 0; i < 5.; ++i) {
    CPPUNIT_ASSERT( !Andpredicate( Vector(i, 0., 0.)) );
    CPPUNIT_ASSERT( !Andpredicate( Vector(i, 1., 0.)) );
    CPPUNIT_ASSERT( !Andpredicate( Vector(i, -1., 0.)) );
    CPPUNIT_ASSERT( !Andpredicate( Vector(i, 0., 1.)) );
    CPPUNIT_ASSERT( !Andpredicate( Vector(i, 0., -1.)) );
  }
  CPPUNIT_ASSERT( !Andpredicate( Vector(5.,-1.,0.)) );
  CPPUNIT_ASSERT( Andpredicate( Vector(5.,0.,1.)) );
  CPPUNIT_ASSERT( !Andpredicate( Vector(5.,0.,-1.)) );
  CPPUNIT_ASSERT( Andpredicate( Vector(6.,0.,0.)) );
}
/** Test whether operator() returns as desired
 *
 */
void Ops_FillPredicateTest::OrTest()
{
  FillPredicate Orpredicate = (*Domainpredicate) || (*Voidpredicate);

  // in OR case: Voidpredicate will always throw because it internally checks
  // whether Linked Cell structure gets correct index to cell, which fails under
  // Ignore boundary conditions
#ifndef NDEBUG
  std::cout << "The following Assertion is intended and does not represent a failure of the test." << std::endl;
  CPPUNIT_ASSERT_THROW( Orpredicate( Vector(-1.,1.,0.)), Assert::AssertionFailure );
#else
  CPPUNIT_ASSERT( Orpredicate( Vector(-1.,1.,0.)) );
#endif
	CPPUNIT_ASSERT( Orpredicate( Vector(5.,1.,0.)) );
#ifndef NDEBUG
  std::cout << "The following Assertion is intended and does not represent a failure of the test." << std::endl;
  CPPUNIT_ASSERT_THROW( Orpredicate( Vector(-2.,0.,0.)), Assert::AssertionFailure );
#else
  CPPUNIT_ASSERT( Orpredicate( Vector(-2.,0.,0.)) );
#endif
  for (double i = 0; i < 5.; ++i) {
    CPPUNIT_ASSERT( Orpredicate( Vector(i, 0., 0.)) );
    CPPUNIT_ASSERT( Orpredicate( Vector(i, 1., 0.)) );
#ifndef NDEBUG
  std::cout << "The following Assertion is intended and does not represent a failure of the test." << std::endl;
  CPPUNIT_ASSERT_THROW( Orpredicate( Vector(i, -1., 0.)), Assert::AssertionFailure );
#else
  CPPUNIT_ASSERT( !Orpredicate( Vector(i, -1., 0.)) );
#endif
    CPPUNIT_ASSERT( Orpredicate( Vector(i, 0., 1.)) );
#ifndef NDEBUG
  std::cout << "The following Assertion is intended and does not represent a failure of the test." << std::endl;
  CPPUNIT_ASSERT_THROW( Orpredicate( Vector(i, 0., -1.)), Assert::AssertionFailure );
#else
  CPPUNIT_ASSERT( !Orpredicate( Vector(i, 0., -1.)) );
#endif
  }
  CPPUNIT_ASSERT( Orpredicate( Vector(5.,1.,0.)) );
#ifndef NDEBUG
  std::cout << "The following Assertion is intended and does not represent a failure of the test." << std::endl;
  CPPUNIT_ASSERT_THROW( Orpredicate( Vector(5.,-1.,0.)), Assert::AssertionFailure );
#else
  CPPUNIT_ASSERT( Orpredicate( Vector(5.,-1.,0.)) );
#endif
  CPPUNIT_ASSERT( Orpredicate( Vector(5.,0.,1.)) );
#ifndef NDEBUG
  std::cout << "The following Assertion is intended and does not represent a failure of the test." << std::endl;
  CPPUNIT_ASSERT_THROW( Orpredicate( Vector(5.,0.,-1.)), Assert::AssertionFailure );
#else
  CPPUNIT_ASSERT( Orpredicate( Vector(5.,0.,-1.)) );
#endif
  CPPUNIT_ASSERT( Orpredicate( Vector(6.,0.,0.)) );
}

/** Test whether operator!() returns as desired
 *
 */
void Ops_FillPredicateTest::NotTest()
{
  Box &domain = World::getInstance().getDomain();
  Node origin(0.,0.,0.);
  Node center(10.,10.,10.);
  Node corner(20.,20.,20.);
  Node outside(30.,30.,30.);

  FillPredicate Notpredicate = !(*Domainpredicate);
  FillPredicate Notpredicate2 = !(*Voidpredicate);

  domain.setConditions(
      BoundaryConditions::Conditions_t(3, BoundaryConditions::Wrap)
  );
  // boundary conditions: Wrap
  {
    // origin is inside
    CPPUNIT_ASSERT( !Notpredicate(origin) );

    // center is inside
    CPPUNIT_ASSERT( !Notpredicate(center) );

    // corner is inside
    CPPUNIT_ASSERT( !Notpredicate(corner) );

    // outside is outside
    CPPUNIT_ASSERT( !Notpredicate(outside) );
  }

  // boundary conditions: Bounce
  {
    domain.setCondition(0, BoundaryConditions::Bounce);
    // origin is inside
    CPPUNIT_ASSERT( !Notpredicate(origin) );

    // center is inside
    CPPUNIT_ASSERT( !Notpredicate(center) );

    // corner is inside
    CPPUNIT_ASSERT( !Notpredicate(corner) );

    // outside is outside
    CPPUNIT_ASSERT( !Notpredicate(outside) );
    domain.setCondition(0, BoundaryConditions::Wrap);
  }

  // boundary conditions: Ignore
  {
    domain.setCondition(0, BoundaryConditions::Ignore);
    // origin is inside
    CPPUNIT_ASSERT( !Notpredicate(origin) );

    // center is inside
    CPPUNIT_ASSERT( !Notpredicate(center) );

    // corner is inside
    CPPUNIT_ASSERT( !Notpredicate(corner) );

    // outside is outside
    CPPUNIT_ASSERT( Notpredicate(outside) );
    domain.setCondition(0, BoundaryConditions::Wrap);
  }

  // set BCs back to default Ignore
  domain.setConditions(
      BoundaryConditions::Conditions_t(3, BoundaryConditions::Ignore)
  );
	// check on not void
#ifndef NDEBUG
  std::cout << "The following Assertion is intended and does not represent a failure of the test." << std::endl;
  CPPUNIT_ASSERT_THROW( Notpredicate2( Vector(-2.,0.,0.)), Assert::AssertionFailure );
#else
  CPPUNIT_ASSERT( !Notpredicate2( Vector(-2.,0.,0.)) );
#endif
  for (double i = 0; i < 5.; ++i) {
    CPPUNIT_ASSERT( Notpredicate2( Vector(i, 0., 0.)) );
    CPPUNIT_ASSERT( Notpredicate2( Vector(i, 1., 0.)) );
#ifndef NDEBUG
  std::cout << "The following Assertion is intended and does not represent a failure of the test." << std::endl;
  CPPUNIT_ASSERT_THROW( Notpredicate2( Vector(i, -1., 0.)), Assert::AssertionFailure );
#else
  CPPUNIT_ASSERT( Notpredicate2( Vector(i, -1., 0.)) );
#endif
    CPPUNIT_ASSERT( Notpredicate2( Vector(i, 0., 1.)) );
#ifndef NDEBUG
  std::cout << "The following Assertion is intended and does not represent a failure of the test." << std::endl;
  CPPUNIT_ASSERT_THROW( Notpredicate2( Vector(i, 0., -1.)), Assert::AssertionFailure );
#else
  CPPUNIT_ASSERT( Notpredicate2( Vector(i, 0., -1.)) );
#endif
  }
  CPPUNIT_ASSERT( !Notpredicate2( Vector(5.,1.,0.)) );
#ifndef NDEBUG
  std::cout << "The following Assertion is intended and does not represent a failure of the test." << std::endl;
  CPPUNIT_ASSERT_THROW( Notpredicate2( Vector(5.,-1.,0.)), Assert::AssertionFailure );
#else
  CPPUNIT_ASSERT( !Notpredicate2( Vector(5.,-1.,0.)) );
#endif
  CPPUNIT_ASSERT( !Notpredicate2( Vector(5.,0.,1.)) );
#ifndef NDEBUG
  std::cout << "The following Assertion is intended and does not represent a failure of the test." << std::endl;
  CPPUNIT_ASSERT_THROW( Notpredicate2( Vector(5.,0.,-1.)), Assert::AssertionFailure );
#else
  CPPUNIT_ASSERT( !Notpredicate2( Vector(5.,0.,-1.)) );
#endif
  CPPUNIT_ASSERT( !Notpredicate2( Vector(6.,0.,0.)) );
}
