/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * CubeMesh.cpp
 *
 *  Created on: Jan 20, 2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include <boost/lambda/lambda.hpp>
#include <cmath>

#include "CodePatterns/Log.hpp"
#include "CubeMesh.hpp"
#include "LinearAlgebra/RealSpaceMatrix.hpp"
#include "LinearAlgebra/Vector.hpp"

/** Constructor for class CubeMesh.
 *
 * Here, we generate nodes homogeneously distributed over a cuboid.
 *
 * \a offset shifts the coordinates, e.g. if counts = 2, we would set nodes at
 * 0 and 0.5 with offset = 0 and 0.4999... and 0.9999.. with offset = 0.999...
 *
 * @param counts number of points per axis
 * @param offset offset Vector (coordinates in [0,1))
 * @param M matrix to transform the default cuboid from (0,0,0) to (1,1,1).
 */
CubeMesh::CubeMesh(const Vector &counts, const Vector &offset, const RealSpaceMatrix &M)
{
  RealSpaceMatrix partition;
  int n[NDIM];

#ifndef NDEBUG
  for (size_t i=0;i<NDIM;++i) {
    ASSERT((offset[i] >= 0.) && (offset[i] < 1.),
        "CubeMesh::CubeMesh() - offset coordinates must be in [0,1) but offset["
        +toString(i)+"] is "+toString(offset[i])+".");
    ASSERT(counts[i] != 0.,
        "CubeMesh::CubeMesh() - counts["+toString(i)+"] must be != "+toString(counts[i])+".");
  }
#endif

  partition.setZero();
  for(int i=0;i<NDIM;i++)
    partition.at(i,i) =  1./(double)counts[i];
  LOG(1, "INFO: partition is " << partition << ".");


  // go over [0,1]^3 filler grid
  for (n[0] = 0; n[0] < counts[0]; n[0]++)
    for (n[1] = 0; n[1] < counts[1]; n[1]++)
      for (n[2] = 0; n[2] < counts[2]; n[2]++) {
        const Vector CurrentPosition((double)n[0], (double)n[1], (double)n[2]);
        nodes.push_back(M * partition * (CurrentPosition + offset));
      }
}

/** Destructor for class CubeMesh.
 *
 */
CubeMesh::~CubeMesh()
{}
