/*
 * RandomInserter.hpp
 *
 *  Created on: Feb 21, 2012
 *      Author: heber
 */

#ifndef RANDOMINSERTER_HPP_
#define RANDOMINSERTER_HPP_


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "InserterBase.hpp"

#include <vector>

#include "LinearAlgebra/Vector.hpp"

class RandomNumberGenerator;
class RealSpaceMatrix;

/** RandomInserter implements Inserter interface such that the given \a Cluster
 * is inserted at the position with a slight random variation.
 *
 * \warning As the RandomInserter rotates the molecule, the only bounding shape in
 * Cluster that makes sense is a sphere. Otherwise, RandomInserter::Max_Attempts
 * attempts are made at finding a lucky shot where all atoms, though rotated
 * and randomly translated, remain inside but after that the cluster is inserted
 * as it is.
 */
class RandomInserter : public InserterBase
{
public:
  RandomInserter(const double _MaxAtomComponent, const double _MaxMoleculeComponent, const bool _DoRandomRotation);
  ~RandomInserter();

  bool operator()(ClusterInterface::Cluster_impl cluster, const Vector &offset) const;

private:
  bool AreClustersAtomsInside(ClusterInterface::Cluster_impl cluster) const;
  Vector getRandomVector(const double range, const double offset) const;

  void doTranslation(
      ClusterInterface::Cluster_impl cluster,
      const RealSpaceMatrix &Rotations,
      const std::vector<Vector> &RandomAtomTranslations,
      const Vector &RandomMoleculeTranslations) const;
  void undoTranslation(
      ClusterInterface::Cluster_impl cluster,
      const RealSpaceMatrix &Rotations,
      const std::vector<Vector> &RandomAtomTranslations,
      const Vector &RandomMoleculeTranslations) const;

private:
  //!> internal random number generator
  RandomNumberGenerator &random;
  //!> minimum value of the random number distribution
  const double rng_min;
  //!> maximum value of the random number distribution
  const double rng_max;
  //!> range is maximum component for atom translations
  const double MaxAtomComponent;
  //!> range is maximum component for molecule translations
  const double MaxMoleculeComponent;
  //!> whether to randomly rotate or not
  const bool DoRandomRotation;

  //!> maximum number of tries to find a suitable randomized position
  static size_t Max_Attempts;
};


#endif /* RANDOMINSERTER_HPP_ */
