/*
 * ClusterInterface.hpp
 *
 *  Created on: Jan 20, 2012
 *      Author: heber
 */

#ifndef CLUSTERINTERFACE_HPP_
#define CLUSTERINTERFACE_HPP_


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <boost/shared_ptr.hpp>
#include <set>

#include "AtomIdSet.hpp"
#include "types.hpp"

class atom;
class CopyAtomsInterface;
class RealSpaceMatrix;
class Shape;
class Vector;

/** This is the interface class for clusters.
 *
 * Clusters are containers for \refAtomIdSet that make sure that within a given
 * shape each of the atoms in the \ref AtomIdSet is contained and none else.
 *
 * We need this ClusterInterface to allow for ClusterInterface::clone() to not
 * return a pointer but a boost::shared_ptr which is much cleaner in terms of
 * memory management.
 *
 */
class ClusterInterface
{
public:
  //!> typedef for pointer to ClusterInterface-compatible instances
  typedef boost::shared_ptr<ClusterInterface> Cluster_impl;
  typedef AtomIdSet::atomIdSet atomIdSet;

  virtual ~ClusterInterface() {}

  /** Getter for the set of atomic ids associated to the cluster.
   *
   * @return set of atomic ids
   */
  virtual const atomIdSet & getAtomIds() const=0;

  /** Getter for the set of atoms associated to the cluster.
   *
   * @return set of atoms
   */
  AtomIdSet getAtoms() const
  {
    return AtomIdSet(getAtomIds());
  }

  /** Getter for the cluster's Shape.
   *
   * @return Shape \a s
   */
  virtual const Shape& getShape() const=0;

  /** Clones this cluster.
   *
   * We copy the shape and we copy all contained atoms in such a way as
   * functor \a copyMethod does. We also translate atoms by \a offset
   * eventually.
   *
   * @param copyMethod functor that knows how to copy the atoms
   * @param offset translational offset for contained atoms
   * @return reference to the clone cluster
   */
  virtual Cluster_impl clone(CopyAtomsInterface& copyMethod, const Vector &offset) const =0;

  /** Move a cluster.
   *
   * This moves both all contained atoms associated to the cluster and the
   * shape itself
   *
   * @param offset translation vector
   */
  virtual void translate(const Vector &offset)=0;

  /** Transforms a cluster.
   *
   * This transforms both all contained atoms associated to the cluster and the
   * shape itself with respect to their spatial positions.
   *
   * @param M transformation matrix
   */
  virtual void transform(const RealSpaceMatrix &M)=0;

  /** Check whether the given atom is inside the Cluster's shape.
   *
   * @param id atom id to check
   * @return true - is inside cluster's shape, false - else
   */
  virtual bool isInside(const atomId_t id) const = 0;
};


#endif /* CLUSTERINTERFACE_HPP_ */
