/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * CopyAtomsInterfaceUnitTest.cpp
 *
 *  Created on: Mar 17, 2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CopyAtomsInterfaceUnitTest.hpp"

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>
#include <iostream>

#include "CodePatterns/Log.hpp"

#include "Atom/atom.hpp"
#include "Atom/CopyAtoms/CopyAtomsInterface.hpp"
#include "Atom/CopyAtoms/CopyAtoms_Simple.hpp"
#include "Atom/CopyAtoms/CopyAtoms_withBonds.hpp"
#include "Bond/bond.hpp"
#include "World.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/
// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( CopyAtomsInterfaceTest );

// set up and tear down
void CopyAtomsInterfaceTest::setUp()
{
  setVerbosity(5);

  for(int i=0;i<ATOM_COUNT;++i){
    atoms[i]= World::getInstance().createAtom();
    atomIds[i]= atoms[i]->getId();
  }
}

void CopyAtomsInterfaceTest::tearDown()
{
  World::purgeInstance();
}

/** Unit test on whether CopyAtomsInterface::operator() works as expected.
 *
 */
void CopyAtomsInterfaceTest::CopyAtomsInterfaceOperatorTest()
{
  CopyAtomsInterface copyMethod;
  CopyAtomsInterface::AtomVector atomVec(atoms, atoms+ATOM_COUNT);

  copyMethod(atomVec);

  CPPUNIT_ASSERT_EQUAL( (size_t)2, World::getInstance().getAllAtoms().size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)2, atomVec.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)2, copyMethod.CopiedAtoms.size() );
  CPPUNIT_ASSERT_EQUAL( (atom *) NULL, copyMethod.CopiedAtoms[0] );
}

/** Unit test on whether CopyAtoms_Simple::operator() works as expected.
 *
 */
void CopyAtomsInterfaceTest::CopyAtoms_SimpleOperatorTest()
{
  CopyAtoms_Simple copyMethod;
  CopyAtomsInterface::AtomVector atomVec(atoms, atoms+ATOM_COUNT);

  copyMethod(atomVec);

  CPPUNIT_ASSERT_EQUAL( (size_t)4, World::getInstance().getAllAtoms().size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)2, atomVec.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)2, copyMethod.CopiedAtoms.size() );
  CPPUNIT_ASSERT( copyMethod.CopiedAtoms[0] != NULL );
}

/** Unit test on whether CopyAtoms_withBonds::operator() works as expected.
 *
 */
void CopyAtomsInterfaceTest::CopyAtoms_withBondsOperatorTest()
{
  CopyAtoms_withBonds copyMethod;
  {
    CopyAtomsInterface::AtomVector atomVec(atoms, atoms+ATOM_COUNT);
    CPPUNIT_ASSERT_EQUAL( (size_t)2, atomVec.size() );

    copyMethod(atomVec);

    CPPUNIT_ASSERT_EQUAL( (size_t)4, World::getInstance().getAllAtoms().size() );
    CPPUNIT_ASSERT_EQUAL( (size_t)2, copyMethod.CopiedAtoms.size() );
    CPPUNIT_ASSERT( copyMethod.CopiedAtoms[0] != NULL );
    CPPUNIT_ASSERT( copyMethod.CopiedAtoms[1] != NULL );
    CPPUNIT_ASSERT_EQUAL( copyMethod.CopiedAtoms, copyMethod.getCopiedAtoms() );
  }

  // create bonds
  {
    atoms[0]->addBond( (size_t)0, atoms[1]);
    const BondList & ListOfBonds = atoms[0]->getListOfBonds();
    CPPUNIT_ASSERT( !ListOfBonds.empty() );
    const bond * const _bond = *ListOfBonds.begin();
    CPPUNIT_ASSERT( _bond->GetOtherAtom(atoms[0]) == atoms[1] );
  }
  {
    CopyAtomsInterface::AtomVector atomVec(atoms, atoms+ATOM_COUNT);
    CPPUNIT_ASSERT_EQUAL( (size_t)2, atomVec.size() );

    copyMethod(atomVec);

    CPPUNIT_ASSERT_EQUAL( (size_t)6, World::getInstance().getAllAtoms().size() );
    CPPUNIT_ASSERT_EQUAL( (size_t)2, copyMethod.CopiedAtoms.size() );
    CPPUNIT_ASSERT( copyMethod.CopiedAtoms[0] != NULL );
    CPPUNIT_ASSERT( copyMethod.CopiedAtoms[1] != NULL );
    const BondList & ListOfBonds = copyMethod.CopiedAtoms[0]->getListOfBonds();
    CPPUNIT_ASSERT( !ListOfBonds.empty() );
    const bond * const _bond = *ListOfBonds.begin();
    CPPUNIT_ASSERT( _bond->GetOtherAtom(copyMethod.CopiedAtoms[0]) == copyMethod.CopiedAtoms[1] );
  }
}


