/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * AllAtomsInsideCuboidAction.cpp
 *
 *  Created on: Aug 9, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "Descriptors/AtomShapeDescriptor.hpp"
#include "Descriptors/AtomSelectionDescriptor.hpp"
#include "Atom/atom.hpp"
#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"
#include "LinearAlgebra/RealSpaceMatrix.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "Shapes/BaseShapes.hpp"
#include "Shapes/Shape.hpp"
#include "Shapes/ShapeOps.hpp"
#include "World.hpp"

#include <boost/foreach.hpp>
#include <iostream>
#include <string>

#include "AllAtomsInsideCuboidAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "AllAtomsInsideCuboidAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
Action::state_ptr SelectionAllAtomsInsideCuboidAction::performCall() {
  RealSpaceMatrix RotationMatrix;

  // obtain information
  getParametersfromValueStorage();

  RotationMatrix.setRotation(params.Xangle, params.Yangle, params.Zangle);

  LOG(1, "Selecting all atoms inside a rotated " << RotationMatrix << " cuboid at " << params.position << " and extension of " << params.extension << ".");
  Shape s = translate(transform(stretch(Cuboid(),params.extension),RotationMatrix),params.position);
  std::vector<atom *> selectedAtoms = World::getInstance().getAllAtoms(AtomsBySelection() && AtomsByShape(s));
  World::getInstance().selectAllAtoms(AtomsByShape(s));
  LOG(0, World::getInstance().countSelectedAtoms() << " atoms selected.");
  return Action::state_ptr(new SelectionAllAtomsInsideCuboidState(selectedAtoms, s, params));
}

Action::state_ptr SelectionAllAtomsInsideCuboidAction::performUndo(Action::state_ptr _state) {
  SelectionAllAtomsInsideCuboidState *state = assert_cast<SelectionAllAtomsInsideCuboidState*>(_state.get());

  World::getInstance().unselectAllAtoms(AtomsByShape(state->s));
  BOOST_FOREACH(atom *_atom, state->selectedAtoms)
    World::getInstance().selectAtom(_atom);

  return Action::state_ptr(_state);
}

Action::state_ptr SelectionAllAtomsInsideCuboidAction::performRedo(Action::state_ptr _state){
  SelectionAllAtomsInsideCuboidState *state = assert_cast<SelectionAllAtomsInsideCuboidState*>(_state.get());
  RealSpaceMatrix RotationMatrix;

  World::getInstance().selectAllAtoms(AtomsByShape(state->s));

  return Action::state_ptr(_state);
}

bool SelectionAllAtomsInsideCuboidAction::canUndo() {
  return true;
}

bool SelectionAllAtomsInsideCuboidAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
