/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * CenterInBoxAction.cpp
 *
 *  Created on: May 8, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// include headers that implement a archive in simple text format
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>
#include "boost/serialization/vector.hpp"

#include "CodePatterns/MemDebug.hpp"

#include <boost/shared_ptr.hpp>

#include "Box.hpp"
#include "CodePatterns/Log.hpp"
#include "LinearAlgebra/MatrixContent.hpp"
#include "LinearAlgebra/RealSpaceMatrix.hpp"
#include "molecule.hpp"
#include "World.hpp"

#include <iostream>
#include <string>
#include <vector>

#include "Actions/MoleculeAction/CenterInBoxAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "CenterInBoxAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
ActionState::ptr MoleculeCenterInBoxAction::performCall() {
  // create undo state
  std::stringstream undostream;
  boost::archive::text_oarchive oa(undostream);
  const RealSpaceMatrix &matrix = World::getInstance().getDomain().getM();
  oa << matrix;
  std::vector< boost::shared_ptr<Vector> > OldPositions;
  {
    std::vector<const molecule*> SelectedMolecules = const_cast<const World &>(World::getInstance()).
        getSelectedMolecules();
    for (std::vector<const molecule*>::iterator MolRunner = SelectedMolecules.begin();
        MolRunner != SelectedMolecules.end();
        ++MolRunner) {
      for(molecule::const_iterator AtomRunner = (*MolRunner)->begin();
          AtomRunner != (*MolRunner)->end();
          ++AtomRunner) {
        OldPositions.push_back(
            boost::shared_ptr<Vector>(new Vector(
                (*AtomRunner)->getPosition()
                ))
            );
      }
    }
  }

  // set new domain
  World::getInstance().setDomain(params.cell_size.get());

  // center atoms
  std::vector<molecule*> SelectedMolecules = World::getInstance().getSelectedMolecules();
  for (std::vector<molecule*>::iterator MolRunner = SelectedMolecules.begin(); MolRunner != SelectedMolecules.end(); ++MolRunner) {
    (*MolRunner)->CenterInBox();
  }

  // give final box size
  LOG(0, "Box domain is now " << World::getInstance().getDomain().getM());

  // create undo state
  MoleculeCenterInBoxState *UndoState =
      new MoleculeCenterInBoxState(
          undostream.str(),
          OldPositions,
          params
          );

  return ActionState::ptr(UndoState);
}

ActionState::ptr MoleculeCenterInBoxAction::performUndo(ActionState::ptr _state) {
  MoleculeCenterInBoxState *state = assert_cast<MoleculeCenterInBoxState*>(_state.get());

  // restore domain
  RealSpaceMatrix matrix;
  std::stringstream undostream(state->undostring);
  boost::archive::text_iarchive ia(undostream);
  ia >> matrix;
  World::getInstance().setDomain(matrix);

  // place atoms on old positions
  std::vector< boost::shared_ptr<Vector> >::const_iterator OldPositionsIter = state->OldPositions.begin();
  std::vector<molecule*> SelectedMolecules = World::getInstance().getSelectedMolecules();
  for (std::vector<molecule*>::iterator MolRunner = SelectedMolecules.begin();
      MolRunner != SelectedMolecules.end();
      ++MolRunner) {
    for(molecule::iterator AtomRunner = (*MolRunner)->begin();
        AtomRunner != (*MolRunner)->end();
        ++AtomRunner) {
      ASSERT(OldPositionsIter != state->OldPositions.end(),
          "WorldBoundInBoxAction::performUndo() - too few positions stored in UndoState.");
      (*AtomRunner)->setPosition(**(OldPositionsIter++));
    }
  }

  // give final box size
  LOG(0, "Box domain restored to " << World::getInstance().getDomain().getM());

  return ActionState::ptr(_state);
}

ActionState::ptr MoleculeCenterInBoxAction::performRedo(ActionState::ptr _state){
  MoleculeCenterInBoxState *state = assert_cast<MoleculeCenterInBoxState*>(_state.get());

  // set new domain
  World::getInstance().setDomain(state->params.cell_size.get());

  // center atoms
  std::vector<molecule *> SelectedMolecules = World::getInstance().getSelectedMolecules();
  for (std::vector<molecule*>::iterator MolRunner = SelectedMolecules.begin(); MolRunner != SelectedMolecules.end(); ++MolRunner) {
    (*MolRunner)->CenterInBox();
  }

  // give final box size
  LOG(0, "Box domain is again " << World::getInstance().getDomain().getM());

  return ActionState::ptr(_state);
}

bool MoleculeCenterInBoxAction::canUndo() {
  return true;
}

bool MoleculeCenterInBoxAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
